// CSMACA.cpp: implementation of the CSMACA class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ProtocalSim.h"
#include "CSMACA.h"
#include "NetworkStats.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CSMACA::CSMACA()
{
	m_DIFSDefered = false;

}

CSMACA::~CSMACA()
{

}

//===================================================
// Run()
//
// Called to simulate it. Should be called by the network
// each run
//
// This function runs until the simulate time is equal to the number
// of ticks allocated to a single run cycle.  This is so that
// if there is no traffic this function could just be run once a cycle
// or every time if there is a lot of traffic
//===================================================
void CSMACA::Run()
{
	// while we still can do operations before giving up control
	while (m_CurrBitTick < m_CurrBitQuanta)
	{
		// if we have something to send make it so
		if (m_bPacketToSend)
		{
			m_DIFSDefered = false; // we are not in defered mode
			int BytesLeft = (m_CurrBitQuanta - m_CurrBitTick)/8;
			// send the current packet
			if ((m_nCurrPacketSize - m_nCurrPacket) > BytesLeft)
			{
				// only send some of it
				m_nCurrPacket += BytesLeft;
				m_CurrBitTick = m_CurrBitQuanta;
				g_NetworkStats.PhyLayerStats.brAllFrames +=BytesLeft*8;
				ASSERT( m_CurrBitTick <= m_CurrBitQuanta);
				// calculate the clock
				m_USecondsTicks += (m_CurrBitTick-m_LastBitTick)/DIFFUTICKSBITRATE;
				m_LastBitTick = m_CurrBitTick;		
			}
			else
			{
				m_bPacketToSend = false;
				m_CurrBitTick += (m_nCurrPacketSize - m_nCurrPacket)*8;
				ASSERT( m_CurrBitTick <= m_CurrBitQuanta);
				g_NetworkStats.PhyLayerStats.brAllFrames += (m_nCurrPacketSize - m_nCurrPacket)*8;
				// calculate the clock
				m_USecondsTicks += (m_CurrBitTick-m_LastBitTick)/DIFFUTICKSBITRATE;
				m_LastBitTick = m_CurrBitTick;				
				SendData(); // actual send the data to the other devices
			}
			
		}		
		else
		{
			if (!GetNextPacket())
			{
				// we don't have anything left to do now, just increment
				// the clock to get out of this loop
				m_CurrBitTick = m_CurrBitQuanta;
				m_USecondsTicks += (m_CurrBitTick-m_LastBitTick)/DIFFUTICKSBITRATE;
				g_NetworkStats.PhyLayerStats.brIdle += (m_CurrBitTick-m_LastBitTick);
				m_LastBitTick = m_CurrBitTick;
			}
//			ASSERT(m_CurrBitTick <= m_CurrBitQuanta);
		}		
	
	}
	//ASSERT(m_CurrBitTick == m_CurrBitQuanta);
	m_CGenStatsTime++;
	// we might have some left overs add to the next loop time
	m_LastBitTick = m_CurrBitTick = m_CurrBitTick - m_CurrBitQuanta; 
	ASSERT(m_LastBitTick >= 0);
	if (m_CGenStatsTime == m_GenStatsTime)
	{
		// generate statistics
		m_StatsNumTimes++;
		GenerateStatistics();
		m_CGenStatsTime = 0;
	}

}

//===================================================
// bool GetNextPacket()
//
// This function gets a packet by looking at all the nics
// If it can't find one for the remaining time period it returns false
//
// Algorithm
//   1) Check if the Access point has something to send within one PIFS
//   2) If not check if anyone has something to send after one DIFS.  Check
//        Per slot if more then one process wants to send go up one slot
//  
// Note: because of the noise model NICs are told that there packet didn't get through
//       and this is only changed as soon as it goes through so we only need to do
//       event handling when something goes through successfully.
//===================================================
bool CSMACA::GetNextPacket()
{

	int PossibleNICs[MAXNIC];
	int NumPossibleNICs = 0;
	bool CollisionOccured = false;


	if (!m_DIFSDefered)
	{
		//=======================================================================//
		// SIFS Check

		// first check if someone has something to send within the SIFS frame, should
		// only be up to one device
		IncrementTime(SIFS); // increment the time by a SIFS
		g_NetworkStats.PhyLayerStats.brIFS += SIFS*DIFFUTICKSBITRATE;

		// check if anyone has anything to send within one SIFS
		for (int ix = 0; ix < m_NumNIC; ix++)
		{
			// k check if any of them have something to send
			int HasPacketToSend =  m_aNIC[ix]->DataToSend(SIFSFrame);
			if (HasPacketToSend != NOMESSAGETOSEND)
			{
				PossibleNICs[NumPossibleNICs] = ix;
				NumPossibleNICs++;	

			}	
		}
		
		ASSERT (NumPossibleNICs <= 1); // should never happen
		if (NumPossibleNICs > 1)
		{	
			m_TotalCollisions++;
			CollisionOccured = true;
		}
		else if(NumPossibleNICs == 1)
		{
			char *pBuffer = m_CurrPacket;
			m_CurrPacketNICIndex = PossibleNICs[0];
			// this call also copies the data over
			bool ValidData = m_aNIC[m_CurrPacketNICIndex]->GetData(pBuffer,m_nCurrPacketSize);				
			m_nCurrPacketSize += PHYPREABLEB;
			ASSERT(ValidData);			
			m_bPacketToSend = true;
			m_nCurrPacket = 0; // the current packet index	
		}

		if (m_bPacketToSend)
			return true;

		IncrementTime(PIFS-SIFS);
		g_NetworkStats.PhyLayerStats.brIFS += (PIFS-SIFS)*DIFFUTICKSBITRATE;
		//=======================================================================//
		// SIFS Check Done Start the PIFS

		if (m_APNIC != NULL)
		{
			int HasPacketToSend =  m_APNIC->DataToSend(PIFSFrame);
			if (HasPacketToSend != NOMESSAGETOSEND)
			{
				char *pBuffer = m_CurrPacket;
				m_CurrPacketNICIndex = PossibleNICs[0];
				// this call also copies the data over
				bool ValidData = m_aNIC[m_CurrPacketNICIndex]->GetData(pBuffer,m_nCurrPacketSize);				
				ASSERT(ValidData);			
				m_bPacketToSend = true;
				m_nCurrPacket = 0; // the current packet index	
			}

			if (m_bPacketToSend)
				return true;
		}
		
		IncrementTime(DIFS-PIFS);	
		g_NetworkStats.PhyLayerStats.brIFS += (DIFS-PIFS)*DIFFUTICKSBITRATE;
		m_DIFSDefered = true; // we are in DIFS mode until we send something
	}

	bool CollisionSoRepeat;
	bool OutofTime = false;

	do
	{
		CollisionSoRepeat = false;
		//=======================================================================//
		// PIFS Check Done Start the DIFS
		// check if anyone has anything to send within one DIFS
		int MaxSlots = (m_CurrBitQuanta - m_CurrBitTick)/(SLOTTIME*DIFFUTICKSBITRATE);
		int LowestFrameCount = 999999;
		for (int ix = 0; ix < m_NumNIC; ix++)
		{
			// k check if any of them have something to send, this will
			// return the frame backup timer
			int HasPacketToSend =  m_aNIC[ix]->DataToSend(DIFSFrame);
			if (HasPacketToSend != NOMESSAGETOSEND)
			{
				if (HasPacketToSend < LowestFrameCount)
				{
					PossibleNICs[0] = ix;
					NumPossibleNICs = 1;
					LowestFrameCount = HasPacketToSend;
				}
				else if (HasPacketToSend == LowestFrameCount)
				{
					PossibleNICs[NumPossibleNICs] = ix;
					NumPossibleNICs++;
				}

			}	
		}
		

		if (NumPossibleNICs == 0)
		{
			// do nothing
		}
		else if (LowestFrameCount > MaxSlots)
		{
			OutofTime = true;
			// defer to the next cycle
			for (ix = 0; ix < m_NumNIC; ix++)
			{
				m_aNIC[ix]->DecrementBackoff(MaxSlots);
			}
		}
		else if (NumPossibleNICs == 1)
		{
			// decrement the backoff time of all of the nics
			for (ix = 0; ix < m_NumNIC; ix++)
			{
				m_aNIC[ix]->DecrementBackoff(LowestFrameCount+1);
			}

			// good one source it has the channel, allow it to send out
			// the data then
			char *pBuffer = m_CurrPacket;
			m_CurrPacketNICIndex = PossibleNICs[0];
			// this call also copies the data over
			bool ValidData = m_aNIC[m_CurrPacketNICIndex]->GetData(pBuffer,m_nCurrPacketSize);				
			ASSERT(ValidData);			
			m_bPacketToSend = true;
			m_nCurrPacket = 0; // the current packet index	
			IncrementTime(SLOTTIME*LowestFrameCount);
			g_NetworkStats.PhyLayerStats.brCollisionFrames += SLOTTIME*LowestFrameCount*DIFFUTICKSBITRATE;
		
		}
		else
		{
			// decrement the backoff time of all of the nics
			for (ix = 0; ix < m_NumNIC; ix++)
			{
				m_aNIC[ix]->DecrementBackoff(LowestFrameCount+1);
			}

			// Get the current packets to signal that they were put unto the medium but
			// they are not going anywhere
			char *pBuffer = m_CurrPacket;
			for (int ix = 0; ix < NumPossibleNICs; ix++)
			{
				m_CurrPacketNICIndex = PossibleNICs[ix];
			// this call also copies the data over
				bool ValidData = m_aNIC[m_CurrPacketNICIndex]->GetData(pBuffer,m_nCurrPacketSize);				
				ASSERT(ValidData);			
			}
			IncrementTime(SLOTTIME*(LowestFrameCount+1));
			g_NetworkStats.PhyLayerStats.brCollisionFrames += SLOTTIME*(LowestFrameCount+1)*DIFFUTICKSBITRATE;
			g_NetworkStats.PhyLayerStats.nFrameCollisions++;
		}

	} while (CollisionSoRepeat && !OutofTime);


	if (!m_bPacketToSend && !CollisionOccured)
	{
		return false;
	}

	return true;

}
