// NIC.cpp: implementation of the NIC class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ProtocalSim.h"
#include "NIC.h"
#include "Network.h"
#include "NetworkStats.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

NIC::NIC()
{
	s_NicStats.CAvgWaitTime = 0;
	s_NicStats.nSentPackets = 0;
	m_FrameBackoff = 0;
	m_ResendTries = 0;		     
	s_NicStats.NumDroppedPackets = 0;
	s_NicStats.CAvgResendTime = 0;
	s_NicStats.NumPackQueueFull = 0;
	m_FramePriority = DIFSFrame;
	m_bSendtheDataNow = false;
	m_bIsAP = false;
	m_DataPackets.b_DataQueue = true;
}

NIC::~NIC()
{


}

//=============================================================
// bool EnterData(char *Buffer, int &nBufferLen)
//
// Enters the data into the packets, returns true if it was entered successfully
//=============================================================
bool NIC::EnterData(char *Buffer, int &nBufferLen)
{
	int tReturn = m_DataPackets.AddData(Buffer,nBufferLen);
	
	if (tReturn == QUEUEFULL)
	{
		s_NicStats.NumPackQueueFull++;
		return false;
	}

	// if there is nothing in the queue then reset the time etc.
	if (tReturn == QFIRSTENTRY)
	{		
		m_FrameBackoff = GetBackoffFrames(0);
		m_PacketReadyToSendTime = NGetUTicks();		
		m_ResendTries = 0;
	}

	return true;

}


//=============================================================
// bool DataToSend
//
// Called by a physical layer to see if we ahve something to send
//
// returns -ve to + MINISLOT time then send, otherwise not time yet
//=============================================================
int NIC::DataToSend(InterFraceSpaces FrameType)
{
	if (FrameType == SIFSFrame)
	{
		if (m_FramePriority == SIFSFrame)
		{
			return 0;
		}
	}
	else if (FrameType == PIFSFrame)
	{
		if (m_FramePriority == PIFSFrame)
		{
			return 0;
		}
	}
	else if (FrameType == DIFSFrame)
	{
		if (!m_DataPackets.QueueEmpty())
		{
			ASSERT(m_FrameBackoff >= 0);
			return m_FrameBackoff;
		}
	}

	return NOMESSAGETOSEND; // a very long time so this is always skipping
}

//=============================================================
// bool GetData(int MaxBytes)
//
// Called by a physical layer to allow the nic to send data
//
// If so it gives the physical layer a packet to send as that is
// the discrete unit
//
//=============================================================
bool NIC::GetData(char *&pBuffer, int &nBufferLen)
{
	if (m_FramePriority == SIFSFrame)
	{
		if (m_bSendtheDataNow)
		{
			m_bSendtheDataNow = false;
			// we just got a CTS packet so send the data now
			memcpy(pBuffer,m_PacketToSend,m_PacketToSendSize*sizeof(char));
			nBufferLen = m_PacketToSendSize;
					
		}
		else
		{			
			// send the high priority frame
			ASSERT(!m_ControlPackets.QueueEmpty());
			m_ControlPackets.GetData(pBuffer,nBufferLen);				
			
		}
		m_FramePriority = DIFSFrame; // packet sent message I guess
		return true;
	}
	else if (m_FramePriority == PIFSFrame)
	{
		// send the high priority frame
		ASSERT(!m_ControlPackets.QueueEmpty());
		m_ControlPackets.GetData(pBuffer,nBufferLen);	
		m_FramePriority = DIFSFrame; // packet sent message I guess
		return true;
	}

	if (m_DataPackets.QueueEmpty())
		return false; 
	
	char *ptBuffer;
	int NumBytes;

	LANINFO LanInfo;
	m_DataPackets.PeakData(ptBuffer,NumBytes);
	g_Protocals.L0GetInfo(ptBuffer,LanInfo);

	// we have the channel if we are here form up a rts packet
	char tBuffer[1000];
	tBuffer[0] = '\0';
	ptBuffer = tBuffer;
	g_Protocals.L0CatRTSInfo(pBuffer,LanInfo,m_bIsAP);
	nBufferLen = g_Protocals.L0GetRTSSize();
	memcpy(pBuffer,tBuffer,nBufferLen);

	// get the data ready, do this here
	if (m_ResendTries < MAXRESEND)
	{
		char *ptBuffer = m_PacketToSend;
		m_DataPackets.GetData(ptBuffer,m_PacketToSendSize,true);
	}
	else
	{
		// we have reached the end, increment the packet queue, if it
		// doesn't get resent it gets lost

		// tell the OS that we have "sent" the packet so it can send the next
		// one if going on event based system
		IPINFO IPInfo;
		char *tBuffer = m_PacketToSend;
		g_Protocals.L0GetInfo(tBuffer,LanInfo);
		g_Protocals.L1GetInfo(tBuffer,IPInfo);
		m_pOS->OnPacketSent(LanInfo,IPInfo);

		char *ptBuffer = m_PacketToSend;
		m_DataPackets.GetData(ptBuffer,m_PacketToSendSize);
	}
	
	m_ResendTries++;
	m_FrameBackoff = GetBackoffFrames(m_ResendTries);




/*

	m_ResendTries++;
	if (m_ResendTries >= MAXRESEND)
	{
		// remove it this time, if it doesn't send correctly is just doesn't go
		m_DataPackets.GetData(pBuffer,nBufferLen,true);
	}
	else
		m_DataPackets.GetData(pBuffer,nBufferLen,false);	
	
*/
	return true;
	
}

//=============================================================
// void ConfirmPacketSent(int Socket, int Port)
//
// After a packet has been sent from the physical layer it calls this function
//=============================================================
void NIC::ConfirmPacketSent(LANINFO LanInfo, IPINFO IPInfo)
{
	// after a packet has been sent if the NIC has another one ready to go, its wait time
	// starts now
	s_NicStats.nSentPackets++;
	m_DataPackets.AdvanceQueue();
	// -1 because it is incremented right after it is sent
	g_NetworkStats.PhyLayerStats.nNumResends += (m_ResendTries-1);
	m_ResendTries = 0;
	m_FrameBackoff = GetBackoffFrames(m_ResendTries);
	m_pOS->OnPacketSent(LanInfo,IPInfo);
	QueueNextPacket();
	//m_NextAvailableTime = NGetUTicks();
	
//	m_PacketReadyToSendTime = m_NextAvailableTime;	
	
}


//=============================================================
// void ConfirmPacketCollision(int Socket, int Port)
//
// When a collision occured this function is called by the physical layer
// not used right now
//=============================================================
void NIC::ConfirmPacketCollision()
{
	m_ResendTries++;
	if (m_ResendTries == MAXRESEND)
	{
		// kill the packet;		
	/*	s_NicStats.NumDroppedPackets++;	
		// find out the socket so we can tell the OS
		char *tPtr = m_aPackets[m_StartQueue];
		LANINFO LanInfo;
		IPINFO IPInfo;
		g_Protocals.L0GetInfo(tPtr,LanInfo);
		g_Protocals.L1GetInfo(tPtr,IPInfo);
		m_pOS->OnPacketSent(IPInfo.SourceIP,IPInfo.Port,MESSAGEDROPPED);
		QueueNextPacket();	
//		m_NextAvailableTime = NGetUTicks();
	//	m_PacketReadyToSendTime = m_NextAvailableTime;	*/
	}
	else
	{
		// reset resend time
		int k = m_ResendTries;
		if (k > 5)
			k = 5;
		int rInterval = 10;
		for (int ix = 1; ix <= k; ix++)
			rInterval *= 2;
		int rTime = rInterval*rand()/(RAND_MAX+1);
//		m_NextAvailableTime = NGetUTicks() + rTime*MINISLOTTIME;
	}
}

//=============================================================
// void QueueNextPacket()
//
// Internal util function to queue up the next packet
//=============================================================
void NIC::QueueNextPacket()
{
/*	m_ResendTries = 0;	
	m_StartQueue = m_StartQueue + 1;
	if (m_StartQueue >= MAXNUMPACKETS)
		m_StartQueue = 0;*/
}

//=============================================================
// void GetStats(NICSTATS &NicStats)
//
// Gets the stats
//=============================================================
void NIC::GetStats(NIC::NICSTATS &NicStats)
{
	if (s_NicStats.nSentPackets == 0)
	{
		NicStats.AvgResendTime = 0;
		NicStats.AvgWaitTime = 0;
	}
	else
	{
		NicStats.AvgWaitTime = (int)(s_NicStats.CAvgWaitTime/(float)s_NicStats.nSentPackets);
		NicStats.AvgResendTime = (float)s_NicStats.CAvgResendTime/(float)s_NicStats.nSentPackets;
	}
	NicStats.NumDroppedPackets = s_NicStats.NumDroppedPackets;
	NicStats.nSentPackets = s_NicStats.nSentPackets;
	NicStats.NumPackQueueFull = s_NicStats.NumPackQueueFull;
}

//=============================================================
// void SendData(char *Buffer, int nBufferLen)
//
// Called by the physical layer when it has data to send to the NIC from another
// device
//
// The NIC should do any processing and operations such as sending an ack
// and send the data to the OS
//=============================================================
void NIC::SendData(char *Buffer, int nBufferLen)
{
	LANINFO LanInfo;
	char *tpBuffer = Buffer;	
	g_Protocals.L0GetInfo(Buffer,LanInfo);
	char *pDataBuffer = Buffer;
	// alright check out the type of packet

	bool AddressedtoMe = false;
	if (m_bIsAP)
	{
		if (LanInfo.DestMac == m_MacAddress)
		{
			AddressedtoMe = true;
		}
	}
	else
	{
		if (LanInfo.DestMac == m_MacAddress)
			AddressedtoMe = true;
	}

	if (LanInfo.FrameControl == RTSFrame)
	{
		if (AddressedtoMe)
		{
			char tBuffer[1000];
			tBuffer[0] = '\0';
			char *pBuffer = tBuffer;
			g_Protocals.L0GetRTSInfo(tpBuffer,LanInfo);		
			g_Protocals.L0CatCTSInfo(pBuffer,LanInfo);
			m_ControlPackets.AddData(tBuffer,g_Protocals.L0GetCTSSize());
			m_FramePriority = SIFSFrame;
		}
	}
	else if (LanInfo.FrameControl == CTSFrame)
	{
		if (AddressedtoMe)
		{
			// allow the data to be sent
			m_bSendtheDataNow = true;
			m_FramePriority = SIFSFrame;
		}
	}
	else if (LanInfo.FrameControl == DataFrame)
	{		
		LANINFO PrevLanInfo = LanInfo;
		if (AddressedtoMe)
		{			
			IPINFO IPInfo;
			g_Protocals.L1GetInfo(Buffer,IPInfo);

			char tBuffer[1000];
			tBuffer[0] = '\0';
			// get the ip and the port for the ack packet
			char *pBuffer = tBuffer;
			g_Protocals.L0CatACKInfo(pBuffer,LanInfo);
			m_ControlPackets.AddData(tBuffer,g_Protocals.L0GetACKSize());
			m_FramePriority = SIFSFrame;
		}
		if (((PrevLanInfo.DSFlags == FROMDS) && (PrevLanInfo.LanType == BroadcastType)) ||
			 ((PrevLanInfo.DSFlags == FROMDS) && (PrevLanInfo.TrueDestMac == m_MacAddress)) ||
			 (m_bIsAP &&  (PrevLanInfo.TrueDestMac == m_MacAddress)))
		{
			// its a broadcast message enter it I guess
			nBufferLen -= g_Protocals.L0GetHeaderSize();
			m_pOS->SendData(pDataBuffer,nBufferLen);
		}

		int blah = FROMDS;
		int asdf = PrevLanInfo.DSFlags == FROMDS;
		int we = PrevLanInfo.LanType == BroadcastType;
		int tr = asdf && we;
		// if we are the access point we should add this to the data queue
		if (m_bIsAP && PrevLanInfo.DSFlags == TODS && (PrevLanInfo.TrueDestMac != m_MacAddress))
		{
			// we need to get the message ready to send out
			// swap the laninfo
			LanInfo = PrevLanInfo;
			LanInfo.DSFlags = FROMDS;
			LanInfo.FrameControl = DataFrame;
			if (LanInfo.TrueDestMac == BROADCASTIP)
			{
				// send back to source for handshaking		
				MACTYPE tMAC = LanInfo.SourceMac;
				LanInfo.SourceMac = LanInfo.DestMac;
				LanInfo.DestMac = tMAC;
			}
			else
			{
				// send to real dest for handshaking
				LanInfo.SourceMac = m_MacAddress;
				LanInfo.DestMac = LanInfo.TrueDestMac;				
			}
			char tPacket[MAXPACKETSIZE];
			tPacket[0] = '\0';
			char *tpPacket = tPacket;
			g_Protocals.L0CatInfo(tpPacket,LanInfo);
			int FullSize = nBufferLen;
			nBufferLen -= g_Protocals.L0GetHeaderSize();
			memcpy(tpPacket,pDataBuffer,nBufferLen); // copy old packet to new one
			m_DataPackets.AddData(tPacket,FullSize);

		}
	}
	else if (LanInfo.FrameControl == ACKFrame)
	{
		// confirm that the data was sent
		if (AddressedtoMe)
		{
			IPINFO IPInfo;
			g_Protocals.L0GetACKInfo(tpBuffer,LanInfo);
			g_Protocals.L1GetInfo(tpBuffer,IPInfo);
			ConfirmPacketSent(LanInfo,IPInfo);
			m_FramePriority = DIFSFrame;
		}
	}

}

//=============================================================
// void DecrementBackoff(int Frames)
//
// Called by the physical layer to decrement the backoff timers a certain
// number of frames
//=============================================================
void NIC::DecrementBackoff(int Frames)
{
	m_FrameBackoff -= Frames;
	// this can happen because all of the times including
	// if this nic is about to send if backedoff
	if (m_FrameBackoff < 0)
		m_FrameBackoff = 0;


}

//=============================================================
// void GetBackoffFrames(int Resends)
//
// Helper function
// Returns the number of frames to backoff given a number of resend attempts
//=============================================================
int NIC::GetBackoffFrames(int Resends)
{
	int Num = m_ResendTries+3;
	int Time = 1;
	for (int ix = 0; ix < Num; ix++)	
		Time = Time * 2;

	int BackoffTime = (int)((float)Time*(float)rand()/RAND_MAX);
	
	ASSERT(BackoffTime >= 0);
	return BackoffTime;

//	return 10*rand()/RAND_MAX;

}
