// OS.cpp: implementation of the OS class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ProtocalSim.h"
#include "OS.h"
#include "Network.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

OS::OS()
{
	for (int ix = 0; ix < MAXSOCKET; ix++)
	{
		m_pSockets[ix] = NULL;
	}
	m_NIC = new NIC;
	m_NIC->m_pOS = this;

}

OS::~OS()
{
	for (int ix = 0; ix < MAXSOCKET; ix++)
	{
		if (m_pSockets[ix] != NULL)
		{
			delete m_pSockets[ix];
		}
	}
	delete m_NIC;
}

//=====================================================================
// int CreateSocket(int PortNumber)
//
// Creates a socket, returns a socket number or -1 if can't create one
//=====================================================================
int OS::CreateSocket(int PortNumber, SOCKETTYPE SocketType, Process *pProcess)
{
	for (int ix = 0; ix < MAXSOCKET; ix++)
	{
		bool Create = false;
		if (m_pSockets[ix] == NULL)
		{
			// create the socket
			m_pSockets[ix] = new SOCKET;
			Create = true;
		}
		else
		{
			if (m_pSockets[ix]->Free)
				Create = true;
		}

		if (Create)
		{
			m_pSockets[ix]->Free = false;
			m_pSockets[ix]->IPInfo.SourceIP = m_IPAddress;		
			m_pSockets[ix]->IPInfo.SocketType = SocketType;
			m_pSockets[ix]->IPInfo.Port = PortNumber;
			m_pSockets[ix]->pProcess = pProcess;
			return ix;
		}

	}
	return -1;
}

//=============================================================
// bool Send(int Socket, char *lpBuf, int nBufLen)
//
// Called by a process to send bytes
//
//=============================================================
bool OS::Send(int Socket, char *lpBuf, int nBufLen)
{
	if (Socket < 0 || Socket >= MAXSOCKET)
		return false;

	if (m_pSockets[Socket] == NULL)
		return false;

	if (m_pSockets[Socket]->Free)
		return false;

	if (nBufLen > MAXMESSAGESIZE)
		return false;

		// ok packetize the thing
	Packetizer.LoadPackets(lpBuf,nBufLen,m_pSockets[Socket]->IPInfo);

	int tBytes;
	char *tPtr;
	while (Packetizer.GetNextPacket(tPtr,tBytes))
	{
		m_NIC->EnterData(tPtr,tBytes);
	}

	return true;
}

//=============================================================
// bool SendTo(int Socket, int DestIP, char *lpBuf, int nBufLen)
//
// Called by a process to send bytes
//
//=============================================================
bool OS::SendTo(int Socket, int DestIP, char *lpBuf, int nBufLen)
{
	if (Socket < 0 || Socket >= MAXSOCKET)
		return false;

	if (m_pSockets[Socket] == NULL)
		return false;

	if (m_pSockets[Socket]->Free)
		return false;

	if (nBufLen > MAXMESSAGESIZE)
		return false;

		// ok packetize the thing
	m_pSockets[Socket]->IPInfo.DestIP = DestIP;
	Packetizer.LoadPackets(lpBuf,nBufLen,m_pSockets[Socket]->IPInfo);


	int tBytes;
	char *tPtr;
	while (Packetizer.GetNextPacket(tPtr,tBytes))
	{
		m_NIC->EnterData(tPtr,tBytes);
	}

	return true;

}

//=============================================================
// void SetSockOpt(int Socket, char *lpBuf, int nBufLen)
//
// Set the socket options
//
//=============================================================
void OS::SetSockOpt(int Socket, int Option)
{
	if (Socket < 0 || Socket >= MAXSOCKET)
		return;

	if (m_pSockets[Socket] == NULL)
		return;

	if (m_pSockets[Socket]->Free)
		return;

	// check if socket is valid
	if (Option | BROADCAST)
	{
		// check if datagram type
		if (m_pSockets[Socket]->IPInfo.SocketType == SockDataGram)
		{
			m_pSockets[Socket]->IPInfo.DestIP = BROADCASTIP;
		}
	}

}

//=============================================================
// void Init(char *&ProcessName)
//
// Inits the OS
//
//=============================================================
void OS::Init(char *&ProcessName, bool m_bIsAccessPoint)
{
	char *tPtr = ProcessName;
	if (m_bIsAccessPoint)
	{
		m_IPAddress = ACCESSPOINTIP; // fixed why not
	}
	else
		m_IPAddress = pNetwork->DNSGetIPAddress(tPtr);	
	m_NIC->m_MacAddress = m_IPAddress; // the same for now
	ASSERT(m_IPAddress != -1);	
	if (m_bIsAccessPoint)
	{
		pPhysicalLayer->RegisterNIC(m_NIC);	
		m_NIC->m_bIsAP = m_bIsAccessPoint;
	}
	else
		pPhysicalLayer->RegisterAP(m_NIC);
}

//=============================================================
// GetStats(OSSTATS &OSStats)
//
// Get the stats
//
//=============================================================
void OS::GetStats(OS::OSSTATS &OSStats)
{
	NIC::NICSTATS NStats;
	m_NIC->GetStats(NStats);
	OSStats.n_AvgResendTime = NStats.AvgResendTime;
	OSStats.n_AvgWaitTime = NStats.AvgWaitTime;
	OSStats.n_nSentPackets = NStats.nSentPackets;
	OSStats.n_NumDroppedPackets = NStats.NumDroppedPackets;
	OSStats.n_NumPackQueueFull = NStats.NumPackQueueFull;

}

//=============================================================
// SendData(char *Buffer, int &nBytes)
//
// Called by the NIC when data is ready
//
//=============================================================
void OS::SendData(char *Buffer, int &nBytes)
{
	IPINFO IPInfo;
	g_Protocals.L1GetInfo(Buffer,IPInfo);
	nBytes -= g_Protocals.L1GetHeaderSize();

	// now match the port to the socket
	for (int ix = 0; ix < MAXSOCKET; ix++)
	{
		if (m_pSockets[ix] != NULL)
		{
			if (!m_pSockets[ix]->Free)
			{
				// k match up the port
				if (m_pSockets[ix]->IPInfo.SourceIP != IPInfo.SourceIP)
				{
					if (m_pSockets[ix]->IPInfo.Port == IPInfo.Port)
					{
					
						// returns true if we have a complete message
						if (m_pSockets[ix]->Buffer.AddData(Buffer,nBytes,IPInfo))
						{
							if (m_pSockets[ix]->pProcess != NULL)
								m_pSockets[ix]->pProcess->OnMessageReceived(IPInfo.SourceIP, IPInfo.Port);
						}
						break;
					}
				}
			}
		}
	}
					


}

//=============================================================
// OnPacketSent(char *Buffer, int &nBytes)
//
// Called by the NIC when data has been sent
//
//=============================================================
void OS::OnPacketSent(LANINFO LanInfo,IPINFO IPInfo)
{
	// if it isn't the last packet in the sequence just quit out
	if (!IPInfo.bLastPacket)
		return;

	for (int ix = 0; ix < MAXSOCKET; ix++)
	{
		if (m_pSockets[ix] != NULL)
		{
			if (!m_pSockets[ix]->Free)
			{
				// k match up the port
				if (m_pSockets[ix]->IPInfo.Port == IPInfo.Port)
				{
					if (m_pSockets[ix]->pProcess != NULL)
						m_pSockets[ix]->pProcess->OnMessageSent(IPInfo.Port);

					break;
				}
			}
		}
	}

}

//=============================================================
// void Receive(int Socket, char &lpBuf, int nBufLen)
//
// Gets data from the incoming buffer if avialable
//
//=============================================================
int OS::Receive(int Socket, char *lpBuf, int nBufLen)
{
	if (!SocketIsValid(Socket))
		return 0;

	int nBytes = m_pSockets[Socket]->Buffer.GetData(lpBuf,nBufLen);
	// the the null for the heck of it
	if (nBytes+1 < nBufLen)
		lpBuf[nBytes] = '\0';
	else
		lpBuf[nBufLen-1] = '0';

	return nBytes;


	

}

//=============================================================
// void SocketIsValid(int SockIndex)
//
// returns if the socket is valid
//
//=============================================================
bool OS::SocketIsValid(int SockIndex)
{
	if (SockIndex < 0 || SockIndex >= MAXSOCKET)
		return false;

	if (m_pSockets[SockIndex] == NULL)
		return false;

	if (m_pSockets[SockIndex]->Free)
		return false;

	return true;
}

