% This is a demo of a extended kalman filter using a robot sim.
% After searching for a web for a matlab version of a robot sim using the ekf, I could not find one that was simple (only a few files)
% and that the algorithm follows a paper reference so it is easy to printout the contents of the matrices so see what values are in
% them at a given time
% Hopefully this could help other people as it shows how each matrix is setup, I just hope I did it accuractly although it does appear to work
%
% Jay Kraut, 2007 
%
% It uses a robot that has x,y and theta, and landmarks are based in x,y although recorded using range and bearing
% it only uses two files, this one and ekf.m which does the ekf.
% landmarks and waypoints are editable in this file
%
% This demonstrates visually the actual position, position after ekf and position with only odemetry particularly if
%   the noise causes a bias such 
%
% a good reference is 
% Simultaneous Localization and Mapping: Part 1 by Hugh Durrant-Whyte and Tim Bailey
%
% Most of the kalman setup was done after reading 
% http://dspace.mit.edu/html/1721.1/36832/16-412JSpring2004/NR/rdonlyres/Aeronautics-and-Astronautics/16-412JSpring2004/D809023E-EF94-46B8-BF8E-1B6D1C86EDE4/0/l8_mapping_3_3.pdf
% main page  
% http://dspace.mit.edu/html/1721.1/36832/16-412JSpring2004/OcwWeb/Aeronautics-and-Astronautics/16-412JSpring2004/CourseHome/index.htm
%
% Notes
% To change the Kalman guessed noise, change this is in the ekf.m file
% Putting in an odometrybias makes this far more interesting
% This uses an oversimplied model of uniform noise for the observations (does not change with distance) and the observations always match correctly
% There are possibly some bugs, or perhaps the filter is not tuned correctly in some instants but the implementation looks pretty close.

clear all


 

%set the noise
OdometryNoise = 0.1;
OdometryBias = 0.45; % note 0.5 is no bias this will cause long term errors when it is not 0.5
ObservationNoise = 0.1;
ObservationBias = 0.5;

%run time
RunTime = 700 %run for 500 iterations

%set the landmarks
Landmarks(1,:) = [3,3];
Landmarks(2,:) = [6,2]
Landmarks(3,:) = [3,6];
NumLandmarks = 3;

%waypoints
Waypoints(1,:) = [6,5];
Waypoints(2,:) = [2,5];
Waypoints(3,:) = [7.5,8.2];


%have some paths
Greenpath(1,:) = Waypoints(1,:)

NumWayPoints = 3;
NextWayPoint = 2; % we start at the first one so the next one is the second one

%robot information
RobotPos(1,:) = Waypoints(1,:); %set the robot at the first waypoint
RobotRot = 0*pi/180; % set the rotation
MaxRotMove = pi/40; %this is the max rotation per itration
FullSpeed = 0.1; %speed when going straight
TurningSpeed = 0.03; %speed when turning

count = 1; %simulation counter

%initialize xstate
xstate = zeros(1,3+NumLandmarks*2);
xstate(1:3) = [RobotPos RobotRot ];
for xs = 1:NumLandmarks
    xstate(xs*2+2:xs*2+3) = Landmarks(xs,:);
end

%xstate = [RobotPos RobotRot Landmarks(1,:) Landmarks(2,:)];

%start the estimate at the current position
EstRobotRot = RobotRot;
EstRobotPos = RobotPos; 

%values that do not get the etf
NoisePos = RobotPos;
NoiseRot = RobotRot;

% setup the covariance here
numlandmarks = NumLandmarks;
length = 3+numlandmarks*2;
C = eye(3+numlandmarks*2)/25;

while(count < RunTime)
count = count + 1; %increment timer

%run the sim here

%figure out the desired angle to the next way point
RobotVec = Waypoints(NextWayPoint,:) - RobotPos;
DesAngle = atan2(RobotVec(2),RobotVec(1));
if (DesAngle < 0)
    DesAngle = DesAngle + 2*pi;
end

%move full speed if we are about straight to the next landmark, otherwise move slower and rotate
Rotation = 0;
Movement = FullSpeed;
Delta = RobotRot - DesAngle;
if (abs(Delta) > 0.04)
    if (Delta > 0)
        RobotRot = RobotRot - MaxRotMove;      
        Rotation = -MaxRotMove;
    else        
         RobotRot = RobotRot + MaxRotMove;
         Rotation = MaxRotMove;         
     end    
     Movement = TurningSpeed; %slow down if rotating
    
end

% this is to translate the movement to x,y
[Vec(1),Vec(2)] = rot(Movement,0,RobotRot);
RobotPos = RobotPos + Vec;

%need this otherwise will not render till end
pause(0.03);

%generate some odemetry noise here
noise = OdometryNoise;
noiseBias = OdometryBias; % (a) if this is 0.5 there is no bais, other wise there is pull to one type of error
                  % or another (like consantly reporting overrotation) this is important to do
                  % to verify that the ekf is actually working
noise1 = (rand-noiseBias)*noise;
noise2 = (rand-noiseBias)*noise;
noise3 = (rand-noiseBias)*noise;

%fill out the estimate movement (contains the noise)
xstate(1) = xstate(1) + Vec(1) + noise1;
xstate(2) = xstate(2) + Vec(2) + noise2;
xstate(3) = xstate(3) + Rotation + noise3;

%this is just for reference if we were not using the ekf
NoisePos(1) = NoisePos(1) + Vec(1) + noise1;
NoisePos(2) = NoisePos(2) + Vec(2)  + noise2;
NoiseRot = NoiseRot + Rotation + noise3;

%do the kalman stuff here
deltat = Movement;
landobs = zeros(1,3+2*numlandmarks);
for x = 1:numlandmarks
    %convert the format of the landmark to range and bearing and add the noise
    %figure out the observations
    noise = ObservationNoise;
    noiseBias = ObservationBias; % (b) no bais should be guassian
    noise1 = (rand-noiseBias)*noise;
    noise2 = (rand-noiseBias)*noise;
    LandmarksN(x,1) =  Landmarks(x,1) +noise1;
    LandmarksN(x,2) =  Landmarks(x,2) +noise2;
    dx= LandmarksN(x,1)  -RobotPos(1); 
    dy= LandmarksN(x,2) -RobotPos(2);
    d2= dx^2 + dy^2;
    d= sqrt(d2);

    landobs(x*2+2:x*2+3)= [d (atan2(dy,dx) - xstate(3))];
end

%k kall the ekf here and then add the delta
[xdelta,C] = ekf(xstate,landobs,deltat,C,numlandmarks);

%correct the state with the delta from the extended kalman filter
xstate = xstate - xdelta;

% in case you want these printed out remove the ';'
EstRobotRot = xstate(3);
EstRobotPos = [xstate(1),xstate(2)];


%check if we reached the next way point if so go to the next one
x1 = RobotPos(1);
y1 = RobotPos(2);
x2 = Waypoints(NextWayPoint,1);
y2 = Waypoints(NextWayPoint,2);
dist = sqrt((x1-x2)^2 + (y1-y2)^2);
if (dist < 0.5)
    NextWayPoint = NextWayPoint + 1;
    if (NextWayPoint == NumWayPoints+1)
        NextWayPoint = 1;
    end
end
    


%draw the robot here (all remaining code is rendering)
size = 0.3; %size of the robot 
rline1 = [0,size];
rline2 = [size/2,-size/2];
rline3 = [-size/2,-size/2];

[rline1(1),rline1(2)] = rot(rline1(1),rline1(2),RobotRot-pi/2);
[rline2(1),rline2(2)] = rot(rline2(1),rline2(2),RobotRot-pi/2);
[rline3(1),rline3(2)] = rot(rline3(1),rline3(2),RobotRot-pi/2);

rline1 = rline1 + RobotPos;
rline2 = rline2 + RobotPos;
rline3 = rline3 + RobotPos;

clf;
hold on

plot([rline2(1),rline3(1)],[rline2(2),rline3(2)],'b')
plot([rline1(1),rline3(1)],[rline1(2),rline3(2)],'b')
plot([rline2(1),rline1(1)],[rline2(2),rline1(2)],'b')

% do the estimated
rline1 = [0,size];
rline2 = [size/2,-size/2];
rline3 = [-size/2,-size/2];

[rline1(1),rline1(2)] = rot(rline1(1),rline1(2),EstRobotRot-pi/2);
[rline2(1),rline2(2)] = rot(rline2(1),rline2(2),EstRobotRot-pi/2);
[rline3(1),rline3(2)] = rot(rline3(1),rline3(2),EstRobotRot-pi/2);

rline1 = rline1 + EstRobotPos;
rline2 = rline2 + EstRobotPos;
rline3 = rline3 + EstRobotPos;


plot([rline2(1),rline3(1)],[rline2(2),rline3(2)],'r')
plot([rline1(1),rline3(1)],[rline1(2),rline3(2)],'r')
plot([rline2(1),rline1(1)],[rline2(2),rline1(2)],'r')

%do the noisy version
rline1 = [0,size];
rline2 = [size/2,-size/2];
rline3 = [-size/2,-size/2];

[rline1(1),rline1(2)] = rot(rline1(1),rline1(2),NoiseRot-pi/2);
[rline2(1),rline2(2)] = rot(rline2(1),rline2(2),NoiseRot-pi/2);
[rline3(1),rline3(2)] = rot(rline3(1),rline3(2),NoiseRot-pi/2);

rline1 = rline1 + NoisePos;
rline2 = rline2 + NoisePos;
rline3 = rline3 + NoisePos;


if count > 1
Greenpath(count-1,:) = NoisePos;
Bluepath(count-1,:) = RobotPos;
Redpath(count-1,:) = EstRobotPos;
end





plot([rline2(1),rline3(1)],[rline2(2),rline3(2)],'g')
plot([rline1(1),rline3(1)],[rline1(2),rline3(2)],'g')
plot([rline2(1),rline1(1)],[rline2(2),rline1(2)],'g')


plot(Greenpath(:,1),Greenpath(:,2),'g-');
plot(Bluepath(:,1),Bluepath(:,2),'b-');
plot(Redpath(:,1),Redpath(:,2),'r-');

plot(Waypoints(:,1),Waypoints(:,2),'x');

plot(LandmarksN(:,1),LandmarksN(:,2),'o');
axis([0,10,0,10]); %force the plot to a certain size
%do the plotting here

text(.5,9,'Noisy Poset','Color','green');
text(.5,8.5,'Actual Poset','Color','blue');
text(.5,8,'ETF Poset','Color','red');

hold off

if count == 2
    pause(2);
end

end
