// AIPath.cpp: implementation of the AIPath class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Simulator.h"
#include "AIPath.h"
#include "ChildView.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

#ifdef VISUALC
extern CChildView *GlobalPtr;
AIPath theAIPath;
#endif

AIPath::AIPath()
{
	// create the path (which is fixed) here
	NumofLinks= 4;
	RedPen.CreatePen(PS_SOLID,1,RGB(255,0,0));
	GreenPen.CreatePen(PS_SOLID,1,RGB(0,255,0));
	// note these values are stored in cm and each link represents one point.
	Links[0].x = 60;
	Links[0].y = 60;
	Links[1].x = 200;
	Links[1].y = 200;
	Links[2].x = 60;
	Links[2].y = 300;
	Links[3].x = 120;
	Links[3].y = 200;
	IterationDistance = 30; // 30 cm iteration distance
	CurrentLink = 0; //always 0
	MaxFoward = 12; // for the curcorners to make the robot goto each point.
	CutCorners = false;

}

AIPath::~AIPath()
{

}

#ifdef VISUALC
/************************************************************************
* Function GDIRender(CDC &MemDC, CRect Quad, CRect Screen)
*
* PURPOSE
* Draw the path on the screen
* INPUT
* MemDC, the GDI device context
* Quad, the area to rendering
* Screen, the screen size
* USAGE
* When Rendering
*************************************************************************/
void AIPath::GDIRender(CDC &MemDC, FRect Quad, CRect Screen, float SRScale, FPoint RenderModelCenter)
{	
    // do the 2D transforms manually here 
    float XScale = (float)Screen.Width()/(float)Quad.Width();
    float YScale = (float)Screen.Height()/(float)Quad.Height();
	 // the links are in mm, rendering is in cm
    float OldXScale = XScale;
    float OldYScale = YScale;
	CPoint Point1,Point2;



    float Scale =  SRScale;// to adjust the size
    XScale = XScale/Scale;
    YScale = YScale/Scale;

    // tranform e.i where to start the lines
    float XTransform = Quad.left;
    float YTransform = Quad.top;
	
	CPen *OldPen = MemDC.SelectObject(&RedPen);

	Point1.x = (long)((Links[0].x - XTransform)*XScale);
	Point1.y = (long)((Links[0].y - YTransform)*YScale);

	MemDC.MoveTo(Point1);

     // fix a GDI scaling problem
     //LAUtils.ClipLine(Screen,XStart,YStart,XEnd,YEnd);
    
	for (int ix = 1; ix < NumofLinks; ix++)
	{
		if ((ix - 1) == CurrentLink)
			MemDC.SelectObject(GreenPen);


		Point2.x = (long)((Links[ix].x - XTransform)*XScale);
		Point2.y = (long)((Links[ix].y - YTransform)*YScale);	
        MemDC.LineTo(Point2);

		if ((ix - 1) == CurrentLink)
			MemDC.SelectObject(RedPen);
	}
	if ((ix - 1) == CurrentLink)
			MemDC.SelectObject(GreenPen);
	MemDC.LineTo(Point1);
    MemDC.SelectObject(OldPen);






}
/************************************************************************
* DirectXRender(VERTEX *&pVertex, FPoint RenderModelCenter, int &VertexCount)
*
* PURPOSE
* DirectX rendering of the screen
*************************************************************************/
void AIPath::DirectXRender(VERTEX *&pVertex, FPoint RenderModelCenter, int &VertexCount)
{

  	float StartHeight = 3;
	float YDistance = 4;
	FPoint Point1,Point2;
	bool CurrentLink1;

	StartPointEnum();

	while(EnumPathPoints(Point1,CurrentLink1));

	for (int ix = 0; ix < NumofLinks; ix++)
	{
		if (ix != NumofLinks -1)
		{
			Point1 = Links[ix];
			Point2 = Links[ix+1];

		}
		else
		{
			Point1 = Links[ix];
			Point2 = Links[0];
		}	

		DWORD Color1 = 0x50ff0000;
		DWORD Color2 = 0x50ff0000;

		if (ix == CurrentLink)
		{
			Color1 = 0x5000ff00;
		    Color2 = 0x5000ff00;
		}


		 // top
		pVertex[0].position.y = StartHeight;
		pVertex[0].position.x = Point1.x;
		pVertex[0].position.z = Point1.y;
		pVertex[0].color = Color1; 
    
		pVertex[1].position.y = StartHeight+YDistance;
		pVertex[1].position.x = Point1.x;
		pVertex[1].position.z = Point1.y;
		pVertex[1].color = Color1;
    
		pVertex[2].position.y = StartHeight+YDistance;
		pVertex[2].position.x = Point2.x;
		pVertex[2].position.z = Point2.y;
		pVertex[2].color = Color2;

		pVertex +=3;
		VertexCount+=1;

		// top
		pVertex[0].position.y = StartHeight+YDistance;
		pVertex[0].position.x = Point2.x;
		pVertex[0].position.z = Point2.y;
		pVertex[0].color = Color2; 
    
		pVertex[1].position.y = StartHeight;
		pVertex[1].position.x = Point2.x;
		pVertex[1].position.z = Point2.y;
		pVertex[1].color = Color2;
    
		pVertex[2].position.y = StartHeight;
		pVertex[2].position.x = Point1.x;
		pVertex[2].position.z = Point1.y;
		pVertex[2].color = Color1;

		pVertex +=3;
		VertexCount+=1;

	}

}

#endif VISUALC

/************************************************************************
* Function FPoint GetNextPoint()
*
* PURPOSE
* Returns the next point on the path
*
* Algorithm
* 1) Find perpidicular point to robot (closest point)
*     1a) Find the line of the link
*     1b) find the perpindicular line using the current robot position
*     1c) Find the intersection which is the point we want
* 2) Traverse the interationDistance along the path to find the desired point,
*     and return this point.
*************************************************************************/
FPoint AIPath::GetNextPoint()
{
	LineStruct Line1,Line2;
    FPoint start,end;
	FPoint CurrPosition,IntersectPoint;
	FPoint tReturn;

	//1a)
			// this routine find the line 
	if (CurrentLink != NumofLinks-1)
	{
		start = Links[CurrentLink];
		end = Links[CurrentLink+1];

	}
	else
	{
		start = Links[CurrentLink];
		end = Links[0];
	}
	
	Line1 = PointstoLine(start,end);

	// 1b) now find the other line
	CurrPosition = GlobalPtr->GlobalGetPosition();

	Line2 = PointstoPerpLine(CurrPosition,Line1);

	// 1c) now get the intersection
	IntersectPoint = FindIntersection(Line1,Line2);


	// find the distance between this point and the starting point
	float Distance1 = GetDistance(start,IntersectPoint);	// find the total distance of the link
	float Distance2 = GetDistance(start,end);
	Distance1 += IterationDistance;

	if (Distance1 > Distance2)
	{
		// switch links
		float Distance3 = GetDistance(CurrentPoint,end);

		if (CutCorners)
		{
			CurrentLink++;
			if (CurrentLink == NumofLinks)
				CurrentLink = 0;

			GetNextPoint();
		}
		else
		{
			// if we are not cutting corners do not goto the next link
			// until we are at close enough to the next point that we
			// can reach it by going foward once

			// in order not to cut corners
			if (Distance3 < MaxFoward)
			{
				CurrentLink++;
				if (CurrentLink == NumofLinks)
					CurrentLink = 0;
			}
			Distance1 = Distance2; // go to the end of the corner
			tReturn = TraverseLine(Distance1,start,end);

			return tReturn;	
		}
	}
	else
	{
		// now use this to find the new x and y;
		tReturn = TraverseLine(Distance1,start,end);

		return tReturn;
	}
}

////////////////////////////////////////////////////////////////////////////
// LineStruct AIPath::PointstoLine(FPoint start, FPoint end)
// 
// PURPOSE
//    Forms the line structure out of two points (ax + by + c = 0)
//
////////////////////////////////////////////////////////////////////////////
LineStruct AIPath::PointstoLine(FPoint start, FPoint end)
{
	LineStruct Line1;

    if ((end.x-start.x) == 0)
    {
        Line1.B= 0;
        Line1.A = 1;
        Line1.C = -(float)start.x;
    }
    else
    {
        float slope = (float)(end.y-start.y)/(float)(end.x-start.x);
        Line1.B = 1;
        Line1.A = (float)-slope;  
        Line1.C = (float)(-start.y+start.x*slope);
    }

    // at this point it should be a 90 degree line and rounding errors will occur because of it.
    // note this isn't fully tested so it could cause problems
    if (Line1.A > 999999)
    {
        Line1.C = Line1.C/Line1.A;
        Line1.A = 1;
        Line1.B = 0;
    }

	return Line1;

}

////////////////////////////////////////////////////////////////////////////
// LineStruct AIPath::PointstoPerpLine(FPoint Start, LineStruct Slope)
// 
// PURPOSE
//    Creates a perpindicular line out of a line and a point
//
////////////////////////////////////////////////////////////////////////////
LineStruct AIPath::PointstoPerpLine(FPoint Start, LineStruct Slope)
{
	 LineStruct Line;
     
     // horionztal line, want vertical line
    if (Slope.A == 0)
    {
        Line.B= 0;
        Line.A = 1;
        Line.C = -(float)Start.x;

    }    
    else
    {
        float slope = (float)Slope.B/(float)(Slope.A);
        Line.B = 1;
        Line.A = (float)-slope;  
        Line.C = (float)(-Start.y+Start.x*slope);
    }

    return Line;

}

/************************************************************************
* Function FPoint FindIntersection(LineStruct Line1, LineStruct Line2)
* 
* PURPOSE
* To return the point of the intersection of two lines
* ALGORITHM
* Kramer rule in linear algebra
* INPUTS
* Two lines in matrix format
* OUTPUTS
* the point of intersection
*************************************************************************/
FPoint AIPath::FindIntersection(LineStruct Line1, LineStruct Line2)
{
    FPoint rPoint;
    // find the bottom determinate
    #ifdef USEFLOAT
        float DBottom = Line1.A*Line2.B-Line2.A*Line1.B;
    #else
        double DBottom = Line1.A*Line2.B-Line2.A*Line1.B;
    #endif

    // find the x determinate
    rPoint.x = (((-Line1.C)*Line2.B-(-Line2.C)*Line1.B)/DBottom);
    rPoint.y = ((((Line1.A *(-Line2.C)-Line2.A*(-Line1.C))/DBottom)));

    return rPoint;    

}


/************************************************************************
* Function FPoint AIPath::TraverseLine(LineStruct Line, float Distance)
* 
* PURPOSE
* To return the point on a line that is distance away from the starting point
* Algorithm
* 1) Get the slope of the line, and then the angle by using the atan function
* 2) Use the angle and distance to find x and y, make sure they are in the 
*    right direction and then add to the starting point
*************************************************************************/
FPoint AIPath::TraverseLine(float Distance, FPoint StartPoint, FPoint EndPoint)
{
	FPoint NewPoint;
	float Slope =(float)(EndPoint.y-StartPoint.y)/(float)(EndPoint.x-StartPoint.x);

	float Angle = (float)atan2(EndPoint.y-StartPoint.y,EndPoint.x-StartPoint.x);
	//float Angle = (float)atan(Slope); // get the angle in radians

	NewPoint.x = (float)(cos(Angle)*Distance);
	NewPoint.y = (float)(sin(Angle)*Distance);

	/*// the Tan function goes from only 0 to 180 to run ifs statements to make
	// sure we are going in the right direction
	if (EndPoint.x-StartPoint.x > 0)
		if (NewPoint.x < 0)
			NewPoint.x = -NewPoint.x;
	if (EndPoint.x-StartPoint.x < 0)
		if (NewPoint.x > 0)
			NewPoint.x = -NewPoint.x;

	if (EndPoint.y-StartPoint.y > 0)
		if (NewPoint.y < 0)
			NewPoint.y = -NewPoint.y;
	if (EndPoint.y-StartPoint.x < 0)
		if (NewPoint.y > 0)
			NewPoint.y = -NewPoint.y;*/

	// add the starting point
	NewPoint.y += StartPoint.y;
	NewPoint.x += StartPoint.x;

	return NewPoint;


}

////////////////////////////////////////////////////////////////////////////
// void AIPath::SetCurrentPoint(FPoint point, float Angle)
// 
// PURPOSE
//    Should be called every time the before the new correction angle is obtained
//
////////////////////////////////////////////////////////////////////////////
void AIPath::SetCurrentPoint(FPoint point, float Angle)
{
	CurrentPoint = point;
	CurrentAngle = Angle;

}


////////////////////////////////////////////////////////////////////////////
// int GetNewDelta()
// 
// PURPOSE
//    This function gets the Delta for steering as per ergenstadts algorithm
//
// Algorithm
//	  1) Its calls a routine GetNextPoint() which gets the next point of the virtual
//		  robot (set distance d from perpindicular point)
//    2) It gets the angle of the robot of the path then subtracts this with 
//        the current angle of the robot, this is the delta the robot has to turn  
//
////////////////////////////////////////////////////////////////////////////
float AIPath::GetNewDelta()
{
	// 1) Gets the next point to travel too.
	FPoint nextPoint = GetNextPoint();

	// get the angle between the two points
	float Angle;
	float dx = nextPoint.x - CurrentPoint.x;
	float dy = nextPoint.y - CurrentPoint.y;

	// if dx = 0; we don't want division by 0.
	if (dx == 0)
	{
		//either 90 or 270
		if (dy > 0)
			Angle = 90;
		else
			Angle = 270;
	}
	else
	{
		float slope = dy/dx;
		Angle = (float)(atan(slope)*180.0/PI); //90 offset
		// tan function doesn't give the right value for quadrent 2 and 3
		if (dx < 0) 
		{
			Angle += 180;
		}
		// keep angle positive
		if (Angle < 0)
			Angle +=360;
	}	
	
	// 2) Get the delta angle
	float dAngle = Angle - CurrentAngle;

	// always pick the smaller direction to rotate
	if (dAngle > 180)
		dAngle = dAngle - 360;
	if (dAngle < -180)
		dAngle = dAngle + 360;

	return dAngle;
}


////////////////////////////////////////////////////////////////////////////
// float AIPath::GetDistance(FPoint One, FPoint Two)
// 
// PURPOSE
//    Gets the distance between two points
//
////////////////////////////////////////////////////////////////////////////
float AIPath::GetDistance(FPoint One, FPoint Two)
{
	return (float)sqrt((double)(One.x-Two.x)*(double)(One.x-Two.x)+(double)(One.y-Two.y)*(double)(One.y-Two.y));


}

////////////////////////////////////////////////////////////////////////////
// bool AIPath::EnumPathPoints(FPoint &returnPoint, bool &IsCurrentLink)
// 
// PURPOSE
//    Used to extract points for rendering
//
////////////////////////////////////////////////////////////////////////////
bool AIPath::EnumPathPoints(FPoint &returnPoint, bool &IsCurrentLink)
{
	if (EnumPathIndex < NumofLinks)
	{
		if (EnumPathIndex ==  CurrentLink)
			IsCurrentLink = true;
		else
			IsCurrentLink = false;

		returnPoint =  Links[EnumPathIndex];
		EnumPathIndex++;
		return true;
	}
	else
		return false;

}

////////////////////////////////////////////////////////////////////////////
// void AIPath::StartPointEnum()
// 
// PURPOSE
//    Starts the enum for rendering
//
////////////////////////////////////////////////////////////////////////////
void AIPath::StartPointEnum()
{
	EnumPathIndex = 0; // start at -1

}

