// AIPathPlanner.cpp: implementation of the AIPathPlanner class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Simulator.h"
#include "AIPathPlanner.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

AIPathPlanner::AIPathPlanner()
{
	ObjectList = NULL; // initiallize
	theQuadTree = NULL;
	GoalQuad = NULL;
	InitialQuad = NULL;
	PathPoints = NULL;
//	Width = 152;
//	Height = 274;
	Width = 220;
	Height = 320;
	GoalPoint.x = 260;
	GoalPoint.y = 100;
	InitialPoint.x = 1;
	InitialPoint.y = 1;
	FPoint tempo;
	tempo.x = 170;
	tempo.y = 100;
	EnumAddObjects(tempo,18);

	tempo.x = 30;
	tempo.y = 80;
	EnumAddObjects(tempo,18);

	tempo.x = 250;
	tempo.y = 20;
	EnumAddObjects(tempo,18);

	GenerateQuadTree();

	GeneratePath();


}

AIPathPlanner::~AIPathPlanner()
{
	Destruct();

}

//====================================================================
// void EnumAddObjects(FPoint Point, float Radius)
//
// Adds objects to an object list that is later used to generate the quadtree
//
// Inputs:
//       Point, the center point of the object
//       Radius, the radius of the objects, all objects are circles
//====================================================================
void AIPathPlanner::EnumAddObjects(FPoint Point, float Radius)
{
	ObjectList1 *temp = new ObjectList1;

	temp->CenterPoint = Point;
	temp->Radius = Radius;
	// link the list.
	temp->Next = ObjectList;
	ObjectList = temp;
}

//====================================================================
// void SetQuadtreeDimensions(float width, float height)
//
// Sets the dimensions of the quadtree
//====================================================================
void AIPathPlanner::SetQuadtreeDimensions(float width, float height)
{
	Width = width;
	Height = height;

}

//====================================================================
// void GenerateQuadTree()
//
// This function initiates the quadtree used to do the search and calls
// the recursive function to generate the tree
//
// Algorithm
// 1) Created big quad
// 2) Initilize the pointer field
// 3) Call recursive function
//====================================================================
void AIPathPlanner::GenerateQuadTree()
{
	// 1) generate the top layers
	FRect Rect;

	Rect.Bottom = 0;
	Rect.Top = Height;
	Rect.Right = 0;
	Rect.Left = Width;

	// 2) initialize pointer table
	for (int ix = 0; ix < FIELDX; ix++)
		for (int iy = 0; iy < FIELDY; iy++)
			PointerTable[ix][iy] = NULL;

	// 3) Call recursive function
	GenerateQuadTree(ObjectList,Rect);


}


//====================================================================
// void GenerateQuadTree(AIPathPlanner::ObjectList1 *Objects, AIPathPlanner::FRect Rect)
//
// This function does a depth style search to make the quadtree
//
// Algorithm
// 1) Check if objects in the objectlist to see if the quadrant
//    is empty, full or mixed
// 2) While checking put any objects that intsect into a new object list
// 3) If full or empty generate quadrant and return
// 4) If mixed call function 4/2 more times with new object list
// 5) If can't go any deeper create mix quadrant
//====================================================================
void AIPathPlanner::GenerateQuadTree(AIPathPlanner::ObjectList1 *Objects, AIPathPlanner::FRect Rect)
{
	// check if quadrant is empty, full or mixed
	ObjectList1 *TraversePtr = Objects;
	ObjectList1 *NewList = NULL;
	QuadTree *tReturn;

	int Count = 0;
	
	// 1) check all of the objects
	while (TraversePtr != NULL)
	{
		// first check if quad is empty first do the circle check
		FPoint QuadCenter;
		QuadCenter.x = Rect.Bottom + (Rect.Top-Rect.Bottom)/2;
		QuadCenter.y = Rect.Right + (Rect.Left-Rect.Right)/2;

		float QuadWidth = Rect.Top-Rect.Bottom;
		float QuadHeight = Rect.Left-Rect.Right;
		float MaxRadius;

		if (QuadWidth > QuadHeight)
			MaxRadius = QuadWidth/2;
		else
			MaxRadius = QuadHeight/2;

		MaxRadius = (float)sqrt((QuadWidth*QuadWidth)/4 + (QuadHeight*QuadHeight)/4);


		
		// rectangle to circle if neccessary
		float Distance = GetDistance(TraversePtr->CenterPoint,QuadCenter);

		bool Intersects = true;

		// first use intersects circles and then actually do the intersects
		if (Distance > (MaxRadius + TraversePtr->Radius))
		{
			// the object is too far off to intersect
			Intersects = false;
			

		}
		else
		{
			Intersects = (DoesIntersect(Rect,TraversePtr->CenterPoint,TraversePtr->Radius));
			// also check if circle center point is in rect
			if ((TraversePtr->CenterPoint.y <= Rect.Left) && (TraversePtr->CenterPoint.y >= Rect.Right))
				if ((TraversePtr->CenterPoint.x <= Rect.Top) && (TraversePtr->CenterPoint.x >= Rect.Bottom))
					Intersects = true;
		}


		// do extra check to make sure the square intersects the circle
		if (!Intersects)
		{
			// 3) do nothing, don't inc the count if still zero then make 
			// an empty quad			

		}
		else
		{

			// next check if it is full by checking if the circle is greater then the quad
			float d1,d2,d3,d4,d;
			FPoint p1,p2,p3,p4;
			p1.x = Rect.Top;
			p1.y = Rect.Left;
			p2.x = Rect.Bottom;
			p2.y = Rect.Left;
			p3.x = Rect.Top;
			p3.y = Rect.Right;
			p4.x = Rect.Bottom;
			p4.y = Rect.Right;
			d1 = GetDistance(TraversePtr->CenterPoint,p1);
			d2 = GetDistance(TraversePtr->CenterPoint,p2);
			d3 = GetDistance(TraversePtr->CenterPoint,p3);
			d4 = GetDistance(TraversePtr->CenterPoint,p4);

			// get the greatest distance 
			d = d1;
			if (d2 > d)
				d = d2;
			if (d3 > d)
				d = d3;
			if (d4 > d)
				d = d4;

			if (TraversePtr->Radius >= d)
			{
			
				// 3) is a full quadrant make count = -9999 to force creation
				Count = -9999;

			}
			else
			{
				// 2) it is a mixed quad, add the object to the object list
				// and increment the count by one

				Count++;
				ObjectList1 *tptr = new ObjectList1;

				tptr->CenterPoint = TraversePtr->CenterPoint;
				tptr->Radius = TraversePtr->Radius;
				tptr->Next = NewList;
				NewList = tptr;		
				
			
			}
		} // end object check
		TraversePtr = TraversePtr->Next;
	} // end traverse objects

	if (Count < 0)
	{
	// 3) it is a full quad, so make it.
		tReturn = CreateQuad(Rect);
		tReturn->QuadStatus = Full;

		// add it to the main list.
		tReturn->Next = theQuadTree;
		theQuadTree = tReturn;	

	}		// it was full an added already
	else if (Count == 0)
	{
		// 3) it is an empty quad, so make it.
		tReturn = CreateQuad(Rect);
		tReturn->QuadStatus = Empty;

		// add it to the main list.
		tReturn->Next = theQuadTree;
		theQuadTree = tReturn;	
	}
	else
	{
		// 4) it must be a mixed quad add it
		//keep on going if we can
		// check if we can keep on going
		if (Rect.Top - Rect.Bottom > DELTAX)
		{
			if (Rect.Left - Rect.Right > DELTAY)
			{
				FRect Quad1, Quad2, Quad3, Quad4;
				// form the new rects

				Quad1.Top = Rect.Top;
				Quad1.Right = Rect.Right;
				Quad1.Left = Rect.Right + (Rect.Left - Rect.Right)/2;
				Quad1.Bottom = Rect.Bottom + (Rect.Top - Rect.Bottom)/2;

				Quad2.Top = Rect.Top;
				Quad2.Right = Rect.Right + (Rect.Left - Rect.Right)/2;
				Quad2.Left = Rect.Left ;
				Quad2.Bottom = Rect.Bottom + (Rect.Top - Rect.Bottom)/2;

				Quad3.Top = Rect.Bottom + (Rect.Top - Rect.Bottom)/2;
				Quad3.Right = Rect.Right + (Rect.Left - Rect.Right)/2;
				Quad3.Left = Rect.Left ;
				Quad3.Bottom = Rect.Bottom;

				Quad4.Top = Rect.Bottom + (Rect.Top - Rect.Bottom)/2;
				Quad4.Right = Rect.Right;
				Quad4.Left = Rect.Right + (Rect.Left - Rect.Right)/2;
				Quad4.Bottom = Rect.Bottom;

				GenerateQuadTree(NewList,Quad1);
				GenerateQuadTree(NewList,Quad2);
				GenerateQuadTree(NewList,Quad3);
				GenerateQuadTree(NewList,Quad4);		
			
			}
			else
			{
				// The height is greater then the width
				// so there is a possiblilty of more quad depth
				// in this dimension
				FRect Left, Right;
				Left = Right = Rect;
				Left.Right = Rect.Right + (Rect.Left - Rect.Right)/2;
				Right.Left = Rect.Right + (Rect.Left - Rect.Right)/2;
				GenerateQuadTree(NewList,Left);
				GenerateQuadTree(NewList,Right);
			}
		}
		else
		{
			// 5) it is an mixed quad, so make it.
			tReturn = CreateQuad(Rect);
			tReturn->QuadStatus = Mixed;



			// add it to the main list.
			tReturn->Next = theQuadTree;
			theQuadTree = tReturn;
		}
	}

	// delete the object list and exit
	ObjectList1 *Prev;				

	while (NewList != NULL)
	{
		Prev = NewList;
		NewList = NewList->Next;
		delete Prev;
	}
	
	


}

//====================================================================
// CreateQuad(AIPathPlanner::FRect Rect)
//
// This function is a helper function for GenterateQuadTree, is creates the
// quad
//
// Note
// The only tricky part is linking the quadrants neighbours together especially 
// since they are different sizes.  To do this I used a pointer table, that stores
// a footprint of the quadrants is able to link to quadrants beside it by looking
// at their footprint
//
//====================================================================
AIPathPlanner::QuadTree* AIPathPlanner::CreateQuad(AIPathPlanner::FRect Rect)
{
	// make the quadrent
	QuadTree *temp = new QuadTree;
	temp->Rect = Rect;
	temp->Next = NULL;
	temp->QuadBottom = NULL;
	temp->QuadRight = NULL;
	temp->QuadLeft = NULL;
	temp->QuadTop = NULL;
	temp->KeyPoint = Normal;
	temp->QuadStatus = Empty;

	int ix,iy;
	QuadTreeList *Linker;
	
	// fill in its footprint in the pointer table
	for (ix = Round(Rect.Bottom/DELTAX); ix < Round(Rect.Top/DELTAX); ix++)
		for (iy = Round(Rect.Right/DELTAY); iy < Round(Rect.Left/DELTAY); iy++)
			PointerTable[ix][iy] = temp;

	// join it with any adjest quads currently in the pointer table

	// join the right quads
	iy = Round(Rect.Right/DELTAY)-1;
	if (iy >= 0)
	{
		for (ix = Round(Rect.Bottom/DELTAX); ix < Round(Rect.Top/DELTAX); ix++)
		{
			if (PointerTable[ix][iy] != NULL)
			{
				// make sure we are only linking once to the adjecent path
				Linker = temp->QuadRight;
				bool Once = false;
				while (Linker != NULL)
				{
					if (Linker->Current == PointerTable[ix][iy])
						Once = true;
					Linker = Linker->Next;
				}

				if (!Once)
				{

					// link the two
					Linker = new QuadTreeList;
					Linker->Current = PointerTable[ix][iy];
					Linker->Next = temp->QuadRight;
					temp->QuadRight = Linker;

					// now link the neighbour to the current one.
					Linker = new QuadTreeList;
					Linker->Current = temp;
					Linker->Next = PointerTable[ix][iy]->QuadLeft;
					PointerTable[ix][iy]->QuadLeft = Linker;
				}
			}
		}
	}

	// join the left quads
	iy = Round(Rect.Left/DELTAY);
	if (iy < FIELDY)
	{
		for (ix = Round(Rect.Bottom/DELTAX); ix < Round(Rect.Top/DELTAX); ix++)
		{
			if (PointerTable[ix][iy] != NULL)
			{

				Linker = temp->QuadLeft;
				bool Once = false;
				while (Linker != NULL)
				{
					if (Linker->Current == PointerTable[ix][iy])
						Once = true;
					Linker = Linker->Next;
				}

				if (!Once)
				{
						// link the two
					Linker = new QuadTreeList;
					Linker->Current = PointerTable[ix][iy];
					Linker->Next = temp->QuadLeft;
					temp->QuadLeft = Linker;

					// now link the neighbour to the current one.
					Linker = new QuadTreeList;
					Linker->Current = temp;
					Linker->Next = PointerTable[ix][iy]->QuadRight;
					PointerTable[ix][iy]->QuadRight = Linker;
				}
			
			}
		}
	}

	// join the top quads
	ix = Round(Rect.Top/DELTAX);
	if (ix < FIELDX)
	{

		for (iy = Round(Rect.Right/DELTAY); iy < Round(Rect.Left/DELTAY); iy++)
		{
			if (PointerTable[ix][iy] != NULL)
			{

				Linker = temp->QuadTop;
				bool Once = false;
				while (Linker != NULL)
				{
					if (Linker->Current == PointerTable[ix][iy])
						Once = true;
					Linker = Linker->Next;
				}

				if (!Once)
				{

					// link the two
					Linker = new QuadTreeList;
					Linker->Current = PointerTable[ix][iy];
					Linker->Next = temp->QuadTop;
					temp->QuadTop = Linker;

					// now link the neighbour to the current one.
					Linker = new QuadTreeList;
					Linker->Current = temp;
					Linker->Next = PointerTable[ix][iy]->QuadBottom;
					PointerTable[ix][iy]->QuadBottom = Linker;
				}
			}
		}
	}

	// join the bottom quads
	ix = Round(Rect.Bottom/DELTAX)-1;
	if (ix >= 0)
	{
		for (iy = Round(Rect.Right/DELTAY); iy < Round(Rect.Left/DELTAY); iy++)
		{
			if (PointerTable[ix][iy] != NULL)
			{

				Linker = temp->QuadBottom;
				bool Once = false;
				while (Linker != NULL)
				{
					if (Linker->Current == PointerTable[ix][iy])
						Once = true;
					Linker = Linker->Next;
				}

				if (!Once)
				{
					// link the two
					Linker = new QuadTreeList;
					Linker->Current = PointerTable[ix][iy];
					Linker->Next = temp->QuadBottom;
					temp->QuadBottom = Linker;

					// now link the neighbour to the current one.
					Linker = new QuadTreeList;
					Linker->Current = temp;
					Linker->Next = PointerTable[ix][iy]->QuadTop;
					PointerTable[ix][iy]->QuadTop = Linker;
				}
			}
		}
	}

	if (PointInRect(InitialPoint,Rect))
	{
		InitialQuad = temp;
		temp->KeyPoint = Initial;
	}

	if (PointInRect(GoalPoint,Rect))
	{
		temp->KeyPoint = Goal;
		GoalQuad = temp;
	}

	
	// compute distances for the A* algorithm
	temp->CenterPoint.x = (Rect.Left - Rect.Right)/2.0f + Rect.Right;
	temp->CenterPoint.y = (Rect.Top - Rect.Bottom)/2.0f + Rect.Bottom;

	// precompute hx because it doesn't change
	temp->hx = GetDistance(temp->CenterPoint,GoalPoint);

	// compute IncDistance for A*
	temp->IncDistance = (Rect.Left - Rect.Right)/2.0f + (Rect.Top - Rect.Bottom)/2.0f;

	return temp;

}

// directX rendering function
void AIPathPlanner::DirectXRender(VERTEX *&pVertex, FPoint RenderModelCenter, int &VertexCount)
{
	QuadTree *Traverse = theQuadTree;
	float QuadHeight = 5.0f;
	DWORD Color,KeyPoint,OnPathColor;
	ObjectList1 *Traverse2 = ObjectList;

	// Color is the default color, however if it is a keypoint (start or end)
	// or a OnPath point different colors are used

	while (Traverse != NULL)
	{

		if (Traverse->QuadStatus == Empty)		
			Color = 0x5000ff00; // green
		else if (Traverse->QuadStatus == Mixed)
			Color = 0x50ffff00; //yellow
		else
			Color = 0x50ff0000; // red

		if (Traverse->KeyPoint == Initial)
			KeyPoint = 0x5040a090;
		else if (Traverse->KeyPoint == Goal)
			KeyPoint = 0x50a0a060;
		else
			KeyPoint = Color;

		if (Traverse->PathInfo == OnPath)
			OnPathColor = 0x50202020;
		else
			OnPathColor = Color;
			

		DWORD Color2 = 0x500000ff; 


		 // one half of the square
		pVertex[0].position.y = QuadHeight;
		pVertex[0].position.z = Traverse->Rect.Top;
		pVertex[0].position.x = Traverse->Rect.Left;
		pVertex[0].color = OnPathColor; 
    
		pVertex[1].position.y = QuadHeight;
		pVertex[1].position.z = Traverse->Rect.Top;
		pVertex[1].position.x = Traverse->Rect.Right;
		pVertex[1].color = OnPathColor;
    
		pVertex[2].position.y = QuadHeight;
		pVertex[2].position.z = Traverse->Rect.Bottom;
		pVertex[2].position.x = Traverse->Rect.Right;
		pVertex[2].color = OnPathColor;

		pVertex +=3;
		VertexCount+=1;

		// one half of the square
		pVertex[0].position.y = QuadHeight;
		pVertex[0].position.z = Traverse->Rect.Bottom;
		pVertex[0].position.x = Traverse->Rect.Right;
		pVertex[0].color = KeyPoint; 
    
		pVertex[1].position.y = QuadHeight;
		pVertex[1].position.z = Traverse->Rect.Bottom;
		pVertex[1].position.x = Traverse->Rect.Left;
		pVertex[1].color = KeyPoint;
    
		pVertex[2].position.y = QuadHeight;
		pVertex[2].position.z = Traverse->Rect.Top;
		pVertex[2].position.x = Traverse->Rect.Left;
		pVertex[2].color = KeyPoint;

		pVertex +=3;
		VertexCount+=1;

		// make one more verticies so we can see the quad
		pVertex[0].position.y = QuadHeight;
		pVertex[0].position.z = Traverse->Rect.Bottom;
		pVertex[0].position.x = Traverse->Rect.Left;
		pVertex[0].color = Color2; 
    
		pVertex[1].position.y = QuadHeight;
		pVertex[1].position.z = Traverse->Rect.Top+.5f;
		pVertex[1].position.x = Traverse->Rect.Right+.5f;
		pVertex[1].color = Color2;
    
		pVertex[2].position.y = QuadHeight+.5f;
		pVertex[2].position.z = Traverse->Rect.Top;
		pVertex[2].position.x = Traverse->Rect.Right;
		pVertex[2].color = Color2;

		pVertex +=3;
		VertexCount+=1;


		Traverse = Traverse->Next;

	}

	while(Traverse2 != NULL)
	{
		
		FPoint PrevPoint,CurrPoint;
		PrevPoint.x = Traverse2->CenterPoint.x ;
		PrevPoint.y = Traverse2->CenterPoint.y+ Traverse2->Radius;
		int numrads = 16;
		float pi = 3.1415f;
		for (int nr = 1; nr <= numrads; nr++)
		{
			CurrPoint.x =  Traverse2->CenterPoint.x + (float)(Traverse2->Radius*sin(2*pi*nr/numrads));
			CurrPoint.y =  Traverse2->CenterPoint.y + (float)(Traverse2->Radius*cos(2*pi*nr/numrads));

			float Objectheight = 2;
					 // one half of the square
			DWORD Color = 0x50af7473;
			pVertex[0].position.y = QuadHeight;
			pVertex[0].position.z = CurrPoint.x;
			pVertex[0].position.x = CurrPoint.y;
			pVertex[0].color = Color; 
    
			pVertex[1].position.y = QuadHeight+Objectheight;
			pVertex[1].position.z = CurrPoint.x;
			pVertex[1].position.x = CurrPoint.y;
			pVertex[1].color = Color;
    
			pVertex[2].position.y = QuadHeight+Objectheight;
			pVertex[2].position.z = PrevPoint.x;
			pVertex[2].position.x = PrevPoint.y;
			pVertex[2].color = Color;

			pVertex +=3;
			VertexCount+=1;

			// one half of the square
			pVertex[0].position.y = QuadHeight+Objectheight;
			pVertex[0].position.z = PrevPoint.x;
			pVertex[0].position.x = PrevPoint.y;
			pVertex[0].color = Color; 
    
			pVertex[1].position.y = QuadHeight;
			pVertex[1].position.z = PrevPoint.x;
			pVertex[1].position.x = PrevPoint.y;
			pVertex[1].color = Color;
    
			pVertex[2].position.y = QuadHeight;
			pVertex[2].position.z = CurrPoint.x;
			pVertex[2].position.x = CurrPoint.y;
			pVertex[2].color = Color;

			pVertex +=3;
			VertexCount+=1;

			PrevPoint = CurrPoint;
		}
		Traverse2 = Traverse2->Next;
	}

		
}


//========================================================================
// DoesIntersect(AIPathPlanner::FRect Rect, FPoint CCenter, float CRadius)
//
// Helper math function returns true if the circle intersects the rectangle
//
//========================================================================
bool AIPathPlanner::DoesIntersect(AIPathPlanner::FRect Rect, FPoint CCenter, float CRadius)
{
	LineStruct Line1,Line2,Line3,Line4;
	FPoint one,two;

	// form the four lines of the rectangle
	one.x = Rect.Top;
	one.y = Rect.Left;
	two.x = Rect.Bottom;
	two.y = Rect.Left;
	Line1 = PointstoLine(one,two);

	one.x = Rect.Top;	
	one.y = Rect.Right;
	two.x = Rect.Bottom;
	two.y = Rect.Right;
	Line2 = PointstoLine(one,two);

	one.x = Rect.Top;
	one.y = Rect.Left;
	two.x = Rect.Top;
	two.y = Rect.Right;
	Line3 = PointstoLine(one,two);

	one.x = Rect.Bottom;
	one.y = Rect.Left;
	two.x = Rect.Bottom;
	two.y = Rect.Right;
	Line4 = PointstoLine(one,two);

	if (DoesIntersect(Line1,CCenter,CRadius))
		return true;
	if (DoesIntersect(Line2,CCenter,CRadius))
		return true;
	if (DoesIntersect(Line3,CCenter,CRadius))
		return true;
	if (DoesIntersect(Line4,CCenter,CRadius))
		return true;

	return false;
	



}

	////////////////////////////////////////////////////////////////////////////
// LineStruct AIPath::PointstoLine(FPoint start, FPoint end)
// 
// PURPOSE
//    Forms the line structure out of two points (ax + by + c = 0)
//
////////////////////////////////////////////////////////////////////////////
LineStruct AIPathPlanner::PointstoLine(FPoint start, FPoint end)
{
	LineStruct Line1;

    if ((end.x-start.x) == 0)
    {
        Line1.B= 0;
        Line1.A = 1;
        Line1.C = -(float)start.x;
    }
    else
    {
        float slope = (float)(end.y-start.y)/(float)(end.x-start.x);
        Line1.B = 1;
        Line1.A = (float)-slope;  
        Line1.C = (float)(-start.y+start.x*slope);
    }

    // at this point it should be a 90 degree line and rounding errors will occur because of it.
    // note this isn't fully tested so it could cause problems
    if (Line1.A > 999999)
    {
        Line1.C = Line1.C/Line1.A;
        Line1.A = 1;
        Line1.B = 0;
    }

	return Line1;
}

// does not work for some reason
#define sqr(in) in*in

//========================================================================
// DoesIntersect(LineStruct Line, FPoint CCenter, float Radius)
//
// Helper math function returns true if the line intersects the circle
//
//========================================================================
bool AIPathPlanner::DoesIntersect(LineStruct Line, FPoint CCenter, float Radius)
{
	float A,B,C;
    if (Line.A == 0)
    {

        A = 1;
        B = (float)(-2*CCenter.x); // used to be negative
        C = CCenter.x*CCenter.x-Radius*(Radius)+(-Line.C - CCenter.y)*(-Line.C - CCenter.y);

		if (4*A*C > B*B)
			return false;

       
    }
    else if (Line.B == 0)
    {

        A = 1;
        B = (float)(2*CCenter.y); // used to be negative
        C =  (CCenter.y)*(CCenter.y)-Radius*(Radius)+(-Line.C - CCenter.x)*(-Line.C - CCenter.x);

		if (4*A*C > B*B)
			return false;
    }
    else
    {
        A = (float)(Line.B*Line.B)/(float)(Line.A*Line.A) + 1;
        B = (float)((2*Line.B/Line.A)*(float)(Line.C/Line.A + CCenter.x) - (float)(2*CCenter.y));
        C = (float)(sqr(Line.C / Line.A + CCenter.x) + sqr((float)CCenter.y) - sqr(Radius));


		if (4*A*C > B*B)
			return false;
    }

	return true;
  


}

//========================================================================
// Destruct()
//
// Destroys any dynamic memory structures
//
//========================================================================
void AIPathPlanner::Destruct()
{
	ObjectList1 *Prev;
	QuadTree *Prev1;
	QuadTreeList *Prev2;
	PathPoints1 *Prev3;

	// delete any outstanding lists
	while (ObjectList != NULL)
	{
		Prev = ObjectList;
		ObjectList = ObjectList->Next;
		delete Prev;
	}

	while (PathPoints != NULL)
	{
		Prev3 = PathPoints;
		PathPoints = PathPoints->Next;
		delete Prev3;
	}

	while (theQuadTree != NULL)
	{
		Prev1 = theQuadTree;
		// take care of the neighbour links

		while (Prev1->QuadBottom != NULL)
		{
			Prev2 = Prev1->QuadBottom;
			Prev1->QuadBottom = Prev1->QuadBottom->Next;
			delete Prev2;
		}
		while (Prev1->QuadLeft != NULL)
		{
			Prev2 = Prev1->QuadLeft;
			Prev1->QuadLeft = Prev1->QuadLeft->Next;
			delete Prev2;
		}
		while (Prev1->QuadRight != NULL)
		{
			Prev2 = Prev1->QuadRight;
			Prev1->QuadRight = Prev1->QuadRight->Next;
			delete Prev2;
		}
		while (Prev1->QuadTop != NULL)
		{
			Prev2 = Prev1->QuadTop;
			Prev1->QuadTop = Prev1->QuadTop->Next;
			delete Prev2;
		}

		theQuadTree = theQuadTree->Next;
		delete Prev1;
	}


}

//====================================================================
// void SetStartGoalPoints(FPoint Start, FPoint Goal)
//
// Sets the start and end points for the path planner
//====================================================================
void AIPathPlanner::SetStartGoalPoints(FPoint Start, FPoint Goal)
{
	InitialPoint = Start;
	GoalPoint = Goal;

}



//====================================================================
// void InitializeSearch()
//
// Does any initialization for the search
//====================================================================
void AIPathPlanner::InitializeSearch()
{
	QuadTree *Traverse = theQuadTree;

	while(Traverse != NULL)
	{
		Traverse->CurrentDistance = 99999.0f; // arbitrarly large
		Traverse->PathInfo = NotonPath;
		Traverse->PathPtr = NULL;
		Traverse = Traverse->Next;
	}

	SearchStackIndex = 0;

}

//====================================================================
// void GeneratePath()
//
// This function generates the path from initial point to goal point
// using iterative deepening with A* heuristics
//
// 1) If we can keep on going go in one direction and add any other directions 
//    to the stack as long as we are bellow the maximum iteration distance
// 2) keep on going until stack is empty if we have not found the end distance
// 3) increment the maximum allowable distance and start again
//====================================================================
void AIPathPlanner::GeneratePath()
{
	float IterativeDepth;
	bool PathFound = false;
	QuadTree *CurrentQuad;	

	IterativeDepth = 30;
	// check interativeDepth to exit eventually
	while (!PathFound && IterativeDepth < 9999.9f)
	{
		IterativeDepth += 20; // deepen the search

		// have to clear everything every time around
		InitializeSearch();
		InitialQuad->CurrentDistance = 0;
		InitialQuad->PathPtr = NULL; // the first point is starting the path
		AddtoStack(InitialQuad);

		while (!StackisEmpty() )
		{
			// get new path;
			CurrentQuad = PopStack();

			if (CurrentQuad->KeyPoint == Goal)
			{
				// we are done enough (due to interative deeping
				// no need to check other paths
				QuadTree *Traverse = CurrentQuad;

				// generate the path
				while (Traverse != NULL)
				{

					Traverse->PathInfo = OnPath;
					Traverse = Traverse->PathPtr;
				}
				// generate the path points
				GeneratePathPoints(CurrentQuad);
				return;
			}

			// check if we should keep on going
			// this is the A* function
			// gx is the current distance
			float fx = CurrentQuad->CurrentDistance + CurrentQuad->hx;

			if (fx < IterativeDepth)
			{
				// k keep on going
				float CurrentDistance = CurrentQuad->CurrentDistance;

				// search the north paths
				QuadTreeList *Traverse;
				Traverse = CurrentQuad->QuadTop;
				SearchPath(Traverse,CurrentQuad,CurrentDistance);

				// search the south paths
				Traverse = CurrentQuad->QuadBottom;
				SearchPath(Traverse,CurrentQuad,CurrentDistance);

				// search the right paths
				Traverse = CurrentQuad->QuadRight;
				SearchPath(Traverse,CurrentQuad,CurrentDistance);

				// search the left paths
				Traverse = CurrentQuad->QuadLeft;
				SearchPath(Traverse,CurrentQuad,CurrentDistance);


			}
		}
	}



}

//====================================================================
// void GeneratePathPoints()
//
// This function, give a link lists of the correct path, generates the
// correct points for the path to travel in.
//
// Algorithm
//   this function uses the overly simplistic pick the middle of the quadrants
//   as the point, may have to be changed.
//
//
//====================================================================
void AIPathPlanner::GeneratePathPoints(AIPathPlanner::QuadTree *PathPtr)
{

	QuadTree *Traverse;
	PathPoints1 *temp;

	// just in case it is not null
	while (PathPoints != NULL)
	{
		temp = PathPoints;
		PathPoints = PathPoints->Next;
		delete temp;
	}

	Traverse = PathPtr;

	while (Traverse != NULL)
	{
		temp = new PathPoints1;
		temp->Points = Traverse->CenterPoint;
		temp->Next = PathPoints;
		PathPoints = temp;
		Traverse = Traverse->PathPtr;
	}

}


//====================================================================
// bool EnumPath(FPoint &point)
//
// This function returns back a point in the path that is generated
// call EnumPathStart to restart the path
//
//====================================================================
bool AIPathPlanner::EnumPath(FPoint &point)
{
	if (TraversePath == NULL)
		return false;

	point = TraversePath->Points;
	TraversePath = TraversePath->Next;

	return true;


}

//====================================================================
// void EnumPathStart()
//
// Restarts the enumeration of the path
//
//====================================================================
void AIPathPlanner::EnumPathStart()
{
	TraversePath = PathPoints;
}
