// Model.cpp: implementation of the Model class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Model.h"
#include "Profiler.h"
#include "binaryformat.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif


// simply macro so it return false saves some code look a bit better
#define getNextToken1( buffer, token,  type) { if (!getNextToken(pBuffer,token,type)) {\
						globalReturnTrue = false; AfxMessageBox("Error from tokanizer"); return false; } }


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Model::Model()
{
	Init();

}

Model::~Model()
{
	Close();

}

/////////////////////////////////////////////////////////////////////
// InitModel(LPDIRECT3DDEVICE8 g_pd3dDevice, const char *fileName)
//
// This function initializes a model and loads it in
//
// It gets a file name and the D3D device
// It first tries to open the file,
// Then it tries to parse it
// If all goes good it load the information from the file into the vertex
// and index buffers
//
// Note: the slowest part is the loading of textures because it filters them into 
// the proper size for the video card.  Until the filtering is done consistently
// externally in terms of resizing this will be a bit on the slow side
//////////////////////////////////////////////////////////////////////
void Model::InitModel(LPDIRECT3DDEVICE8 g_pd3dDevice, const char *fileName)
{
	// first thing to do is to load the file and parse it, and then
	// load it into directx

	theProfiler.EnterStartTime("Profile7");
	theProfiler.EnterStartTime("Profile1");
	CFile loadFile;

	// use the old c file type because the other type does not work with
	// binary files
	FILE *saveFile = fopen(fileName,"rb");

	
	// open the file for reading
/*	if (loadFile.Open(fileName,CFile::typeBinary  | CFile::modeRead | CFile::shareDenyNone ) == 0)
	{
		AfxMessageBox("Cannot open file, name invalid");
		return;
	}*/

	theProfiler.EnterEndTime("Profile1");

	theProfiler.EnterStartTime("Profile2");

	// k now start to parse it, get all of the information out
	char buffer[200000];
//	int nBytesRead = loadFile.Read(buffer,500000);
	int nBytesRead = fread(buffer,sizeof(char),200000,saveFile);
	if (nBytesRead == 200000)
	{
		AfxMessageBox("File size too large\nContact vendor for upgrade");
		return;
	}
	theProfiler.EnterEndTime("Profile2");
	theProfiler.EnterStartTime("Profile3");
	buffer[nBytesRead] = '\0'; // put in an end of file marker
	if (!ParseFile(buffer))
		return;

	theProfiler.EnterEndTime("Profile3");

	theProfiler.EnterStartTime("Profile4");

	// alright now start load everything into memory
	numTotalVertices = 0;
	for (int iy = 0; iy <= currObject; iy++)	
		numTotalVertices += numVertices[iy];

	numTotalFaces = 0;
	for (iy = 0; iy <= currObject; iy++)	
		numTotalFaces += numFaces[iy];


   // allocate the memory
   if( FAILED( g_pd3dDevice->CreateVertexBuffer( numTotalVertices*sizeof(TEXTUREDVERTEXSTRUCT),
												   0 ,
												   TEXTUREDVERTEX,
												   D3DPOOL_DEFAULT,
												   &g_pVertices 
												   ) )) 
                                                   AfxMessageBox("Failed Vertex Buffer");

   if( FAILED( g_pd3dDevice->CreateIndexBuffer( numTotalFaces *3*sizeof(WORD),
                                            D3DUSAGE_WRITEONLY, D3DFMT_INDEX16,
                                            D3DPOOL_DEFAULT, &g_pIndices ) ) )
											AfxMessageBox("Failed Index Buffer");


   // load the textures

    theProfiler.EnterEndTime("Profile4");
	theProfiler.EnterStartTime("Profile5");

	int textureLoads = 0;
	
	for (int iTexture = 0; iTexture <=currObject; iTexture++)
	{
		char tFileName[100];
		char *tPtr = tFileName;
		bool sameTexture = false;
		for (int iRepeatTexture = iTexture - 1; iRepeatTexture >= 0; iRepeatTexture--)
		{
			// check for repeats don't open the same texture twice, later 
			// sort the indices so same textures are aligned in buffer, saves on 
			// calls
			if (textureFileName[iRepeatTexture] != NULL)
			{
				if (!strcmp(textureFileName[iTexture],textureFileName[iRepeatTexture]))
				{
					// k they are the same simply assign the same texture pointer
					m_pModelTexture[iTexture] = m_pModelTexture[iRepeatTexture];
					sameTexture = true;
					break;
				}
			}
		}
		if (!sameTexture)		
		{
			textureLoads++;
			if (textureFileName[iTexture] != NULL)
			{

				// add extra \ to the file name as required
				for (unsigned int ix = 0; (ix < strlen(textureFileName[iTexture])); ix++)
				{
					if (textureFileName[iTexture][ix] == '\\')
					{
						*tPtr = '\\';
						tPtr++;
					}

					*tPtr = textureFileName[iTexture][ix];
					tPtr++;
				}
				*tPtr = '\0';


				//note using triangle filters take a lot of time
			   if( FAILED(D3DXCreateTextureFromFileEx( g_pd3dDevice, tFileName, 
							D3DX_DEFAULT, D3DX_DEFAULT, 1, 0, D3DFMT_R5G6B5, 
							D3DPOOL_MANAGED,  D3DX_FILTER_TRIANGLE  ,/*D3DX_FILTER_TRIANGLE|D3DX_FILTER_MIRROR,*/ 
							D3DX_FILTER_NONE /*D3DX_FILTER_TRIANGLE|D3DX_FILTER_MIRROR*/, 0, NULL, NULL, &m_pModelTexture[iTexture] )))
			   {
				   AfxMessageBox("Error loading texture\n");
			   }
			}
			else
			{
				m_pModelTexture[iTexture] = NULL;
			}
		}
	}

	theProfiler.EnterEndTime("Profile5");

	theProfiler.EnterStartTime("Profile6");

	// now fill vertex and indices buffer
	TEXTUREDVERTEXSTRUCT* pVertices;
	if( FAILED( g_pVertices->Lock( 0, sizeof(TEXTUREDVERTEXSTRUCT)*numTotalVertices, (BYTE**)&pVertices, 0 ) ) )
	{
		AfxMessageBox("CriticalFailure 1");
		return;
	}
	else
	{		
		for (iy = 0; iy <= currObject; iy++)
		{

			if (m_pModelTexture[iy] != NULL)
			{

				for (int ix = 0; ix < numVertices[iy]; ix++)
				{
					pVertices[ix].x = (float)loadVertices[iy][ix].x;
					pVertices[ix].z = (float)loadVertices[iy][ix].z;
					pVertices[ix].y = (float)loadVertices[iy][ix].y;
					pVertices[ix].tu = (float)(loadTextureCoords[iy][ix].x - 1);
					pVertices[ix].tv = (float)(1-loadTextureCoords[iy][ix].y);
					pVertices[ix].color = (DWORD)(0xffffffff);//*((float)rand()/(float)RAND_MAX));
				}
				pVertices += ix;
			}
			else
			{
				
				for (int ix = 0; ix < numVertices[iy]; ix++)
				{
					pVertices[ix].x = (float)loadVertices[iy][ix].x;
					pVertices[ix].z = (float)loadVertices[iy][ix].z;
					pVertices[ix].y = (float)loadVertices[iy][ix].y;
					pVertices[ix].tu = (float)0.0;
					pVertices[ix].tv = (float)0.0;
					pVertices[ix].color = (DWORD)(0xffffffff);//*((float)rand()/(float)RAND_MAX));
				}
				pVertices += ix;
			}

		}
		g_pVertices->Unlock();
	}

	WORD* pIndices;

	if( FAILED( g_pIndices->Lock( 0, 2*numTotalFaces, (BYTE**)&pIndices, 0 ) ) )
	{
		AfxMessageBox("CriticalFailure 2");
		return;
	}
	else
	{
		for (iy = 0; iy <= currObject; iy++)
		{

			for (int ix = 0; ix < numFaces[iy]; ix++)
			{
				pIndices[ix*3] = (unsigned short)loadFaces[iy][ix].x;
				pIndices[ix*3+1] = (unsigned short)loadFaces[iy][ix].y;
				pIndices[ix*3+2] = (unsigned short)loadFaces[iy][ix].z;					
			}
			pIndices += ix*3;
		}

		g_pIndices->Unlock();
	}

	// do some profiling for the heck of it.
	theProfiler.EnterEndTime("Profile6");
	theProfiler.EnterEndTime("Profile7");
	double p1 = theProfiler.GetExecutionTime("Profile1");
	double p2 = theProfiler.GetExecutionTime("Profile2");
	double p3 = theProfiler.GetExecutionTime("Profile3");
	double p4 = theProfiler.GetExecutionTime("Profile4");
	double p5 = theProfiler.GetExecutionTime("Profile5");
	double p6 = theProfiler.GetExecutionTime("Profile6");
	double p7 = theProfiler.GetExecutionTime("Profile7");
	double p8 = p7 - p1 - p2- p3 - p4 - p5 - p6;

	char biggy[400];
	sprintf(biggy,"p1 %7.5f\np2 %7.5f\np3 %7.5f\np4 %7.5f\np5 %7.5f\np6 %7.5f\np7 %7.5f\np8 %7.5f\n",
			p1,p2,p3,p4,p5,p6,p7,p8);
	//AfxMessageBox(biggy);


	modelValidToRender = true;

}

/////////////////////////////////////////////////////////////////////
// Render(LPDIRECT3DDEVICE8 g_pd3dDevice)
//
// Render each object pretty much seperatly swaping textures when required
//
// Most models should only have one texture if done right in 3DS max however
// currently it doesn't reorder the index buffer so simular textures are together
// which can be a future option
//////////////////////////////////////////////////////////////////////
void Model::Render(LPDIRECT3DDEVICE8 g_pd3dDevice)
{

	// only render if we have a loaded model
	if (modelValidToRender)
	{
		// setup the buffers
		g_pd3dDevice->SetStreamSource( 0, g_pVertices, sizeof(TEXTUREDVERTEXSTRUCT) );
		g_pd3dDevice->SetVertexShader( TEXTUREDVERTEX );

	//		g_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_WIREFRAME);	// Wireframe for dubugging
    g_pd3dDevice->SetRenderState(D3DRS_FILLMODE,D3DFILL_SOLID);

	g_pd3dDevice->SetTextureStageState(0,D3DTSS_MAGFILTER ,D3DTEXF_LINEAR);

	// Turn off culling, so we see the front and back of the triangle
    g_pd3dDevice->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );

		g_pd3dDevice->SetIndices( g_pIndices, 0 );

		int totalFaces = 0;
		int totalVertices = 0;

		// render each object as it was loaded in
		for (int iObject = 0; iObject <=  currObject; iObject++)
		{
			// set the texture
			g_pd3dDevice->SetTexture(0,m_pModelTexture[iObject]);
		//	g_pd3dDevice->SetTexture(1,m_pModelTexture[iObject]);

		//	if (iObject == 0)
			g_pd3dDevice->DrawIndexedPrimitive( D3DPT_TRIANGLELIST,totalVertices,
									   numVertices[iObject],totalFaces, numFaces[iObject]);

			totalVertices += numVertices[iObject];
			totalFaces += numFaces[iObject]*3; // *3 because 3 vertices per face

		}
		
		// Null it back to be nice
		g_pd3dDevice->SetTexture(0,NULL);	
	}
		

}

/////////////////////////////////////////////////////////////////////
// bool ParseFile(char *pBuffer)
//
// The start of the RD parser.
//
// This functions looks for the proper header and then keeps on going.
//////////////////////////////////////////////////////////////////////
bool Model::ParseFile(char *pBuffer)
{

	bool isText;
	// parse out the file header information and make sure it is valid
	if (!ParseFileHeader(pBuffer,isText))
		return false;

	if (isText)
	{
		// now we must have an object otherwise quit out
		while (ParseFileObject(pBuffer))
		{
		}
	}
	else
	{
		// now we must have an object otherwise quit out
		while (ParseFileObjectB(pBuffer))
		{
		}
	}

	return globalReturnTrue;

}

//////////////////////////////////////////////////////////////////////////
// bool ParseFileHeader(char *pBuffer)
//
// Part of the RD parser, this function parses out the file header information
//
// returns false if the file format is not valid
//
// isText returns true if it is a text based file, false otherwise
bool Model::ParseFileHeader(char *&pBuffer, bool &isText)
{
	enum tokentype type;
	char token[50];

	getNextToken1(pBuffer,token,type);

	if (type != filedesc)
	{
		AfxMessageBox("Wrong file type abort loading");
		return false;
	}

	getNextToken1(pBuffer,token,type);

	if (type != floatnum)
	{
		AfxMessageBox("Wrong file type abort loading");
		return false;
	} 

	double fileVersion = atof(token);

	if (fileVersion == CURRENTFILEVERSION)
	{
		isText = true;
	}
	else if (fileVersion == CURRENTFILEVERSIONB)
	{
		isText = false;
	}
	else
	{
		AfxMessageBox("Wrong file version number abort loading");
		return false;
	}


	return true;

}

//////////////////////////////////////////////////////////////////////////
// bool ParseFileObject(char *pBuffer)
//
// Part of the RD parser, this function parses out objects
//
// returns false when there are no objects left
bool Model::ParseFileObject(char *&pBuffer)
{
	enum tokentype type;
	char token[50];

	// now get the name of the object out
	getNextToken1(pBuffer,token,type);

	// this return false on a endoffile type which kicks us out of the parsing
	if (type != objectname)
		return false;

	currObject++; // increment the object count

	textureFileName[currObject] = NULL; // in case there isn't one

	char objectName[200];
	objectName[0] = '\0';
	// name cane have spaces go to the brace
	while (type != openbrace)
	{
		strcat(objectName,token);
		getNextToken1(pBuffer,token,type);
	}



	bool done = false;

	while (!done)
	{
		// k now see what kind of information we have
		getNextToken1(pBuffer,token,type);
	
		if (type == geometry)
		{
			if (!ParseFileGeometry(pBuffer))
			{
				globalReturnTrue =false;
				return false;
			}
		}
		else if (type == texturecoords)
		{
			if (!ParseFileTextureInfo(pBuffer))
			{
				globalReturnTrue = false;
				return false;
			}
		}
		else if (type == closebrace)
		{
			done = true;
		}
		else
		{
			// k we don't recognize the key so parse till we hit a closing brace then continue

			getNextToken1(pBuffer,token,type);
			while (type != closebrace)
			{
				getNextToken1(pBuffer,token,type);
			}
		//	AfxMessageBox("Bad info header");
		//	globalReturnTrue = false;
		//	return false;
		}
	}


	return globalReturnTrue; // set if a token goes really wrong


}

//////////////////////////////////////////////////////////////////////////
// bool ParseFileGeometry(char *pBuffer)
//
// Part of the RD parser, this function parses geometry information
// such as vertices and faces
//
// returns false if format is invalid
bool Model::ParseFileGeometry(char *&pBuffer)
{
	char float1[50];
	char float2[50];
	char float3[50];
	char int1[50];
	char int2[50];
	char int3[50];
	enum tokentype type;
	char token[50];

	getNextToken1(pBuffer,token,type); // get rid of the open brace

	if (type != openbrace)
	{
		AfxMessageBox("Invalid open brace in load geometry");
		return false;
	}

	getNextToken1(pBuffer,token,type); // get the number of vertices

	if (type != intnum)
	{
		AfxMessageBox("Invalid vertices number in file loading, aborting");
		return false;
	}

	numVertices[currObject] = atoi(token);

	// get rid of semicolon
	getNextToken1(pBuffer,token,type);


	// allocate the array
	loadVertices[currObject] = new Point3[numVertices[currObject]];
	for (int ix = 0; ix < numVertices[currObject]; ix++)
	{
		getNextToken1(pBuffer,float1,type);
		getNextToken1(pBuffer,token,type);
		getNextToken1(pBuffer,float2,type);
		getNextToken1(pBuffer,token,type);
		getNextToken1(pBuffer,float3,type);
		loadVertices[currObject][ix].x = atof(float1);
		loadVertices[currObject][ix].y = atof(float3);
		loadVertices[currObject][ix].z = -atof(float2);	
		getNextToken1(pBuffer,token,type); // get rid of semicolon
	}

	// k we are at the face number
	getNextToken1(pBuffer,token,type);

	if (type != intnum)
	{
		AfxMessageBox("Invalid vertices number in file loading, aborting");
		return false;
	}

	numFaces[currObject] = atoi(token);
	loadFaces[currObject] = new Point3[numFaces[currObject]];

	getNextToken1(pBuffer,int1,type); // get rid of semicolon

	for (ix = 0; ix < numFaces[currObject]; ix++)
	{
		getNextToken1(pBuffer,int1,type);
		getNextToken1(pBuffer,token,type);
		getNextToken1(pBuffer,int2,type);
		getNextToken1(pBuffer,token,type);
		getNextToken1(pBuffer,int3,type);
		int intt1 = atoi(int1);
		int intt2 = atoi(int2);
		int intt3 = atoi(int3);
		if (intt1 > 50000 || intt1 < 0)
			intt1 = 0;
		if (intt2 > 50000 || intt2 < 0)
			intt2 = 0;
		if (intt3 > 50000 || intt3 < 0)
			intt3 = 0;
		loadFaces[currObject][ix].x = intt1 + currFaces;
		loadFaces[currObject][ix].y = intt2 + currFaces;
		loadFaces[currObject][ix].z = intt3 + currFaces;
		getNextToken1(pBuffer,token,type); // get rid of semicolon
	}

	// get rid of the geometry closing bracket
	getNextToken1(pBuffer,token,type);

	if (type != closebrace)
	{
		AfxMessageBox("bad brace 1");
		return false;
	}
	currFaces += numVertices[currObject]; // so next object load properly

	return true;

}

//////////////////////////////////////////////////////////////////////////
// bool ParseFileTextureInfo(char *pBuffer)
//
// Part of the RD parser, this function parses texture information
// such texture coords and texture file name only one texture per object
// is allowed (by 3ds max)
//
// returns false if format is invalid
bool Model::ParseFileTextureInfo(char *&pBuffer)
{
	char float1[50];
	char float2[50];
	enum tokentype type;
	char token[200];
	
	getNextToken1(pBuffer,token,type);

	// look for the texturecoords
	if (type != openbrace)
	{
		AfxMessageBox("bad open brace 3");
		return false;
	}

	// k first thing is number of texture coords, don't really need it
	// because it is equal to the number of vertices

	getNextToken1(pBuffer,token,type);
	getNextToken1(pBuffer,token,type);

	loadTextureCoords[currObject] = new Point3[numVertices[currObject]];

	for (int ix = 0; ix < numVertices[currObject]; ix++)
	{
		char *tempo = pBuffer;
		getNextToken1(pBuffer,float1,type);
		getNextToken1(pBuffer,token,type);
		getNextToken1(pBuffer,float2,type);
		loadTextureCoords[currObject][ix].x = atof(float1);
		loadTextureCoords[currObject][ix].y = atof(float2);
		getNextToken1(pBuffer,token,type); // get rid of semicolon
	}

	getNextToken1(pBuffer,token,type);

	if (type != texturefilename)
	{
		AfxMessageBox("bad texture file name");
		return false;
	}

	getNextToken1(pBuffer,token,type);

	char objectName[200];
	objectName[0] = '\0';
	// name cane have spaces go to the brace
	while (type != closebrace)
	{
		strcat(objectName,token);
		getNextToken1(pBuffer,token,type);
	}

	textureFileName[currObject] = new char[100];

	// k copy over the texture file name
	strcpy(textureFileName[currObject],objectName);

//	getNextToken1(pBuffer,token,type);

	if (type != closebrace)
	{
		AfxMessageBox("bad brace 2");
		return false;
	}



	return true;
}

//////////////////////////////////////////////////////////////////////////
// Close()
//
// frees all memory
void Model::Close()
{
	for (int ix = 0; ix <= currObject; ix++)
	{
		if (loadVertices[ix] != NULL)
			delete[] loadVertices[ix];
		if (loadFaces[ix] != NULL)
			delete[] loadFaces[ix];
		if (loadTextureCoords[ix] != NULL)
			delete[] loadTextureCoords[ix];
		if (textureFileName[ix] != NULL)
			delete[] textureFileName[ix];

		if( m_pModelTexture[ix] != NULL )
		{
			// go through the list because pointers are shared but
			// we only want to release once, so null duplicates
			for (int iz = ix + 1; iz <= currObject; iz++)
			{
				if (m_pModelTexture[ix] == m_pModelTexture[iz])
					m_pModelTexture[iz] = NULL;
			}
			 m_pModelTexture[ix]->Release();	
		}
	}

	if( g_pIndices != NULL )
		 g_pIndices->Release();

	if( g_pVertices != NULL )
		 g_pVertices->Release();

	modelValidToRender = false;

}

//////////////////////////////////////////////////////////////////////////
// Init()
//
// initializes all memory (NULLS it)
void Model::Init()
{
	currObject = -1;
	numTotalVertices = 0;
	numTotalFaces = 0;
	currFaces = 0;
	modelValidToRender = false;
	globalReturnTrue = true;

	g_pIndices = NULL;
	g_pVertices = NULL;

	for (int ix = 0; ix < MAXOBJECTSPERFILE; ix++)
	{
		numVertices[ix] = NULL;
		numFaces[ix] = NULL;
		loadVertices[ix] = NULL;
		loadFaces[ix] = NULL;
		loadTextureCoords[ix] = NULL;
		textureFileName[ix] = NULL;
		m_pModelTexture[ix] = NULL;
	}


}

//////////////////////////////////////////////////////////////////////////
// Open(LPDIRECT3DDEVICE8 g_pd3dDevice)
//
// Called to open up a file given a file name 
// and the D3D device (neccessary to init memory) 
void Model::Open(LPDIRECT3DDEVICE8 g_pd3dDevice, const char *fileName)
{

	Close();
	Init(); // reset everything
	InitModel(g_pd3dDevice,fileName);
}

//////////////////////////////////////////////////////////////////////////
// bool getNextToken(char *&buffer, char *token, tokentype &type)
//
// The workhorse of all the RD parser this parses out a token in a char stream (buffer)
// and returns the token type and the actual token
//
// Token types are
// enum tokentype {intnum, floatnum, comma, semicolon,geometry,
// objectname,filedesc,openbrace,closebrace,texturecoords,texturefilename,endoffile};
bool Model::getNextToken(char *&buffer, char *token, tokentype &type)
{

	char sep[5];
	sep[0] = ' ';
	sep[1] = '\n';
	sep[2] = 13;
	sep[3] = '\t';
	sep[4] = '\0';

	char filedc[] = "JExporter";
	char geom[] = "Geometry";
	char textureco[] = "TextureInfo";
	char textname[] = "TextureName";


	bool whiteSpace = true;
	while (whiteSpace)
	{
		if (*buffer == '\0')
		{
			type = endoffile;
			return true;
		}
		if (*buffer == sep[0] || *buffer == sep[1] || *buffer == sep[2] || *buffer == sep[3] || *buffer == sep[4])
		{
			buffer++;
		}
		else
			whiteSpace = false;
	}

	if (((*buffer >= 48) && (*buffer <=57)) || (*buffer == '-'))
	{
		// we have a number
		// check if it is a int or a float
		char *tPtr = buffer;
		while ((*tPtr >= 48) && (*tPtr <=57)) 
		{
			if (*tPtr == '\0')
				return false;
			tPtr++;
		}
		if (*tPtr == '.')		
			type = floatnum;
		else
			type = intnum;
		
	}
	else if (*buffer == ',')
		type = comma;
	else if (*buffer == ';')
		type = semicolon;
	else if (*buffer == '{')
		type = openbrace;
	else if (*buffer == '}')
		type = closebrace;
	else
	{
		// must be an indent or a keyword
		char tword[50];
		sscanf(buffer,"%s",tword);
		if (!strcmp(tword,filedc))
			type = filedesc;
		else if (!strcmp(tword,geom))
			type = geometry;
		else if (!strcmp(tword,textureco))
			type = texturecoords;
		else if (!strcmp(tword,textname))
			type = texturefilename;
		else
			type = objectname;
	}

	// don't use sscanf since we can use a strncpy because we know the proper index
/*	char tSccanf = buffer[50];
	buffer[50] = ' ';
	sscanf(buffer,"%s",token);
	buffer[50] = tSccanf;*/

	char *tBuffer = buffer;

	// now go to the next white space;
	int tindex = 0;
	whiteSpace = false;
	while (!whiteSpace)
	{
		if ((*buffer == ',') || (*buffer == ';'))
		{
			if(tindex == 0)
			{
				
				whiteSpace = true;
				token[0] = buffer[0];
				token[1] = '\0';
				buffer++;
			}
			else
			{
				strncpy(token,tBuffer,tindex);
				token[tindex] = '\0';
				whiteSpace = true;
			}
		}
		else if (!(*buffer == sep[0] || *buffer == sep[1] || *buffer == sep[2] || *buffer == sep[3] || *buffer == sep[4] ))
		{
			buffer++;
			tindex++;
		}
		else
		{
			whiteSpace = true;
			strncpy(token,tBuffer,tindex);
			token[tindex] = '\0';
		}
	}
	

	return true;

};



//////////////////////////////////////////////////////////////////////////
// bool ParseFileGeometry(char *pBuffer)
//
// Part of the RD parser, this function parses geometry information
// such as vertices and faces
//
// binary version
//
// returns false if format is invalid
bool Model::ParseFileGeometryB(char *&pBuffer)
{
	float float1,float2,float3;
	WORD int1,int2,int3;
	WORD key;
	DWORD skip;
	// get rid of open brace
	READ_WORD(pBuffer,key);

	// read total number of bytes ignore because we know the key
	READ_DWORD(pBuffer,skip);

	// now figure out how many vertices
	READ_WORD(pBuffer,key);

	numVertices[currObject] = key;

	// allocate the array
	loadVertices[currObject] = new Point3[numVertices[currObject]];
//	float *tempo = new float[numVertices[currObject]*3];
	for (int ix = 0; ix < numVertices[currObject]; ix++)
	{
		READ_FLOAT(pBuffer,float1);
		READ_FLOAT(pBuffer,float2);
		READ_FLOAT(pBuffer,float3);
		ASSERT(float1 < 10000);
		ASSERT(float2 < 10000);
		ASSERT(float3 < 10000);
		loadVertices[currObject][ix].x = float1;
		loadVertices[currObject][ix].y = float3;
		loadVertices[currObject][ix].z = -float2;
	}

	// now figure out how many faces
	READ_WORD(pBuffer,key);

	numFaces[currObject] = key;
	loadFaces[currObject] = new Point3[numFaces[currObject]];	

	for (ix = 0; ix < numFaces[currObject]; ix++)
	{
		READ_WORD(pBuffer,int1);
		READ_WORD(pBuffer,int2);
		READ_WORD(pBuffer,int3);
		loadFaces[currObject][ix].x = int1 + currFaces;
		loadFaces[currObject][ix].y = int2 + currFaces;
		loadFaces[currObject][ix].z = int3 + currFaces;
	}

	// get rid of the geometry closing bracket
	READ_WORD(pBuffer,key);

	if (key != CLOSE_BRACE_KEY)
	{
		AfxMessageBox("bad brace 3");
		return false;
	}
	currFaces += numVertices[currObject]; // so next object load properly

	return true;
}


bool Model::ParseFileObjectB(char *&pBuffer)
{
	char fileName[100];
	WORD key;

	pBuffer++; // get rid of /0

	READ_STRING(pBuffer,fileName); // this should be the object name


	READ_WORD(pBuffer,key);

	if (key != OPEN_BRACE_KEY)
		return false;

	currObject++; // increment the object count

	textureFileName[currObject] = NULL; // in case there isn't one

	// now figure out what kind of data we have

	bool done = false;

	while (!done)
	{
		READ_WORD(pBuffer,key);

		if (key == GEOMETRY_KEY)
		{
			if (!ParseFileGeometryB(pBuffer))
			{
				globalReturnTrue =false;
				return false;
			}
		}
		else if (key == TEXTURE_INFO_KEY)
		{
			if (!ParseFileTextureInfoB(pBuffer))
			{
				globalReturnTrue =false;
				return false;
			}
		}
		else if (key == CLOSE_BRACE_KEY)
		{
			done = true;
		}
		else
		{
			DWORD nBytes;
			READ_WORD(pBuffer,key); // read the open brace
			READ_DWORD(pBuffer,nBytes);
			// k we should skip key bytes then continue
			pBuffer += nBytes;
			READ_WORD(pBuffer,key);
			if (key != CLOSE_BRACE_KEY)
			{
				globalReturnTrue =false;
				return false;
			}
		
		}
	}

	return true;
}

bool Model::ParseFileTextureInfoB(char *&pBuffer)
{
	float float1,float2; 
	WORD key;
	DWORD skip;
	// get rid of open brace
	READ_WORD(pBuffer,key);

	// read total number of bytes ignore because we know the key
	READ_DWORD(pBuffer,skip);

	// now figure out how many vertices, not that we allready don't know
	READ_WORD(pBuffer,key);
	loadTextureCoords[currObject] = new Point3[numVertices[currObject]];

	for (int ix = 0; ix < numVertices[currObject]; ix++)
	{
		READ_FLOAT(pBuffer,float1);
		READ_FLOAT(pBuffer,float2);
		loadTextureCoords[currObject][ix].x = float1;
		loadTextureCoords[currObject][ix].y = float2;
	}

	READ_WORD(pBuffer,key);
	
	if (key != TEXTURE_NAME_KEY)
	{
		AfxMessageBox("bad texture file name");
		return false;
	}

	char tString[100];

	READ_STRING(pBuffer,tString);

	textureFileName[currObject] = new char[100];

	// k copy over the texture file name
	strcpy(textureFileName[currObject],tString);

	// a close brace key
	READ_WORD(pBuffer,key);



	return true;
}
