// OctTree.cpp: implementation of the OctTree class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MapEditor.h"
#include "OctTree.h"
#include "uIDList.h"
#include "TerrainManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

OctTree theOctTree;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

OctTree::OctTree()
{
	
	_Destruct();

	HeadOctPtr = NULL;

	float HeightArray[32];
	HeightArray[0] = 0;
	HeightArray[1] = 1;
	HeightArray[2] = 2;
	HeightArray[3] = 3;
	HeightArray[4] = 4;

	CreateOctTree(4,4,HeightArray,4,4,4);


}

OctTree::~OctTree()
{
	_Destruct();
	GroupIDList.DestructIDList();

}

void OctTree::_Destruct()
{
	theTerrainManager.TerrainInvalid(); // fixs complications
	if (HeadOctPtr != NULL)
	{
			// nuke all of the objects
		OBJECTLIST::iterator curr = HeadOctPtr->ObjectList.begin();
		OBJECTLIST::iterator end = HeadOctPtr->ObjectList.end();
		while (curr != end)
		{
			// delete the object
			delete (*curr);
			curr++;
		} 
		
	
		DeleteOctTree();
		HeadOctPtr = NULL;
	}
	SelectList.clear();
	theTerrainManager.ResetTerrianManager();

}

OctNode::OctNode()
{
	this->Neighbours[0] = NULL;
	this->Neighbours[1] = NULL;
	this->Neighbours[2] = NULL;
	this->Neighbours[3] = NULL;
	this->Neighbours[4] = NULL;
	this->Neighbours[5] = NULL;
	this->Children[0] = NULL;
	this->Children[1] = NULL;
	this->Children[2] = NULL;
	this->Children[3] = NULL;
	this->Children[4] = NULL;
	this->Children[5] = NULL;
	this->Children[6] = NULL;
	this->Children[7] = NULL;
	this->Parent = NULL;
	this->Flag = 0;

}
//---------------------------------------------------------------------
// CreateOctTree()
// 
// Usage: Creates the oct tree
// 
// Inputs:
// float Width: the width of the octree, the z dimension
// float Length: the length of the octree, the x dimension
// float HeightArray[32]: the partioning of the height, the one dimension 
//						  that is not uniform, also the y dimension
// int WidthSegments: the number of width segments, must be a power of 2
// int LengthSegments: the number of length segments, must be a power of 2
// int HeightSegments: the number of height segments, must be a power of 2
//
// Algorithm
// 1) Create all of the nodes
// 2) Connect all of the neighbours
// 1a) is achieved by loading in the node that is to be split into a stack, with information
//	   on how it is to be split further.  This is neccessary because the octree could be unbalanced
//	   in every dimensions.
// 1b) Every stack entry contains a ptr to the node to be split, plus 3 variables that tell
//     at this point how many more times this node is to be split in any of the 3 dimensions
// 1c) The adding to the stack stops when there is no more dimensions to be split (this occurs
//	   simultantiously in all three dimensions).
// 1d) eventually the stack is empty and it stops
// 2a) A look up table is enployed that is used to connect the nieghbours, which replaces
//     a bunch of if statements.
// 2b) The tree is traversed entirely with each node calling a function which connects that
//     node to the neighbours
// Notes: By convention the height is splittable less then the width of length
//---------------------------------------------------------------------
void OctTree::CreateOctTree(float Width, float Length, float HeightArray[], int WidthSegments,
								int LengthSegments, int HeightSegments)
{
	_Destruct();

	// fill in the information in the info structure
	OctreeInfo.Height = HeightArray[HeightSegments];
	OctreeInfo.Length = Length;
	OctreeInfo.Width = Width;
	OctreeInfo.HeightSegments = HeightSegments;
	OctreeInfo.LengthSegments = LengthSegments;
	OctreeInfo.WidthSegments = WidthSegments;


	float WidthofSegments = Width/WidthSegments;
	float LengthofSegments = Length/LengthSegments;
	ConstructOct tConstructOct,tConstructOct2;	

	DeleteOctTree(); // delete the old one

	// make sure we have something to do.
	if (!((WidthSegments >= 1) || (LengthSegments >= 1)))
		return;

	HeadOctPtr = new OctNode;

	// fill in the information of the head ptr
	HeadOctPtr->Area.bX = 0;
	HeadOctPtr->Area.bY = 0;
	HeadOctPtr->Area.bZ = 0;
	HeadOctPtr->Area.tX = Length;
	HeadOctPtr->Area.tY = HeightArray[HeightSegments]-HeightArray[0];
	HeadOctPtr->Area.tZ = Width;
	HeadOctPtr->Parent = NULL;

	//---------------------------------------------------------------
	// If lengthsegments is greater or equal then width segments, everything seems normal
	// If the other way occurs, the way the tree is initially made is slightly different,
	// so some values have to be swapped, to avoid having to duplicate the code, note that
	// the height can never be greater then width or length segments
	tConstructOct.NodePtr = HeadOctPtr;
	int Zero,One,Two,Three;
	if (LengthSegments >= WidthSegments)
	{
		tConstructOct.WidthSplitsLeft = Math.Log2(WidthSegments);
		tConstructOct.LengthSplitsLeft = Math.Log2(LengthSegments);
		Zero = 0;
		One = 1;
		Two = 2;
		Three = 3;
	}
	else
	{
		tConstructOct.WidthSplitsLeft = Math.Log2(LengthSegments);
		tConstructOct.LengthSplitsLeft = Math.Log2(WidthSegments);
		Zero = 0;
		One = 3;
		Two = 1;
		Three = 2;
	}
	tConstructOct.HeightSplitsLeft = Math.Log2(HeightSegments);

	int TotalHeightSplits = Math.Log2(HeightSegments);
	
	ConstructStack.push(tConstructOct);

	while (!ConstructStack.empty())
	{
		 
		tConstructOct = ConstructStack.top();
		ConstructStack.pop(); // pop it out here
		// create as many nodes as possible and then add them to the stack
		// append each node to its neighbours whenever possible
	
		if (tConstructOct.LengthSplitsLeft >= tConstructOct.WidthSplitsLeft)
		{
			// construct the length
			tConstructOct.NodePtr->Children[Zero] = new OctNode;
			tConstructOct.NodePtr->Children[One] = new OctNode;
			tConstructOct.NodePtr->Children[Zero]->Area = tConstructOct.NodePtr->Children[One]->Area = tConstructOct.NodePtr->Area;

			tConstructOct.NodePtr->Children[Zero]->Parent = tConstructOct.NodePtr;
			tConstructOct.NodePtr->Children[One]->Parent = tConstructOct.NodePtr;

			// now split up the area
			if (One == 1) 
			{
				// Length > Width
				tConstructOct.NodePtr->Children[Zero]->Area.bX = tConstructOct.NodePtr->Area.tX - (tConstructOct.NodePtr->Area.tX - tConstructOct.NodePtr->Area.bX)/2.0f;
				tConstructOct.NodePtr->Children[One]->Area.tX = tConstructOct.NodePtr->Area.tX - (tConstructOct.NodePtr->Area.tX - tConstructOct.NodePtr->Area.bX)/2.0f;
			}
			else
			{
				// Width > Length
				tConstructOct.NodePtr->Children[Zero]->Area.tZ = tConstructOct.NodePtr->Area.tZ - (tConstructOct.NodePtr->Area.tZ - tConstructOct.NodePtr->Area.bZ)/2.0f;
				tConstructOct.NodePtr->Children[One]->Area.bZ = tConstructOct.NodePtr->Area.tZ - (tConstructOct.NodePtr->Area.tZ - tConstructOct.NodePtr->Area.bZ)/2.0f;

			}

			if (tConstructOct.LengthSplitsLeft == tConstructOct.WidthSplitsLeft)
			{
				// split in the other dimension also
				// construct the length
				tConstructOct.NodePtr->Children[Two] = new OctNode;
				tConstructOct.NodePtr->Children[Three] = new OctNode;

				tConstructOct.NodePtr->Children[Two]->Parent = tConstructOct.NodePtr;
				tConstructOct.NodePtr->Children[Three]->Parent = tConstructOct.NodePtr;

				float NewMiddle;
				if (One == 1)
				{

					 NewMiddle = tConstructOct.NodePtr->Area.tZ - (tConstructOct.NodePtr->Area.tZ - tConstructOct.NodePtr->Area.bZ)/2.0f;

					tConstructOct.NodePtr->Children[Two]->Area = tConstructOct.NodePtr->Children[One]->Area;
					tConstructOct.NodePtr->Children[Three]->Area = tConstructOct.NodePtr->Children[Zero]->Area;

					// now divide the widths of the area accordingly

					tConstructOct.NodePtr->Children[Zero]->Area.tZ = NewMiddle;
					tConstructOct.NodePtr->Children[One]->Area.tZ = NewMiddle;

					tConstructOct.NodePtr->Children[Two]->Area.bZ = NewMiddle;
					tConstructOct.NodePtr->Children[Three]->Area.bZ = NewMiddle;
				}
				else
				{
					NewMiddle = tConstructOct.NodePtr->Area.tX - (tConstructOct.NodePtr->Area.tX - tConstructOct.NodePtr->Area.bX)/2.0f;

					tConstructOct.NodePtr->Children[Two]->Area = tConstructOct.NodePtr->Children[One]->Area;
					tConstructOct.NodePtr->Children[Three]->Area = tConstructOct.NodePtr->Children[Zero]->Area;

					// now divide the widths of the area accordingly

					tConstructOct.NodePtr->Children[Zero]->Area.bX = NewMiddle;
					tConstructOct.NodePtr->Children[One]->Area.bX = NewMiddle;

					tConstructOct.NodePtr->Children[Two]->Area.tX = NewMiddle;
					tConstructOct.NodePtr->Children[Three]->Area.tX = NewMiddle;


				}

				// now check if the height has to be split up also
				if (tConstructOct.LengthSplitsLeft == tConstructOct.HeightSplitsLeft)
				{
					// the height is a different beast, with the values coorisponding with the
					//height table
					tConstructOct.NodePtr->Children[4] = new OctNode;
					tConstructOct.NodePtr->Children[5] = new OctNode;
					tConstructOct.NodePtr->Children[6] = new OctNode;
					tConstructOct.NodePtr->Children[7] = new OctNode;

					// add reference to their parents
					tConstructOct.NodePtr->Children[4]->Parent = tConstructOct.NodePtr;
					tConstructOct.NodePtr->Children[5]->Parent = tConstructOct.NodePtr;
					tConstructOct.NodePtr->Children[6]->Parent = tConstructOct.NodePtr;
					tConstructOct.NodePtr->Children[7]->Parent = tConstructOct.NodePtr;

					// tricky part, figure out the current heigh to split them
					// by referencing the table
					int top,bottom;
				//	top = HeightSegments;
					for (int ix = 0; ix <= HeightSegments; ix++)
					{
						if (tConstructOct.NodePtr->Area.tY == HeightArray[ix])
						{
							top = ix;
							break;
						}
					}

					bottom = tConstructOct.HeightSplitsLeft;
					bottom = Math.Pwr2(bottom)/2;

					NewMiddle = HeightArray[top - bottom];

					// give the new nodes the correct area
					tConstructOct.NodePtr->Children[4]->Area = tConstructOct.NodePtr->Children[0]->Area;
					tConstructOct.NodePtr->Children[5]->Area = tConstructOct.NodePtr->Children[1]->Area;
					tConstructOct.NodePtr->Children[6]->Area = tConstructOct.NodePtr->Children[2]->Area;
					tConstructOct.NodePtr->Children[7]->Area = tConstructOct.NodePtr->Children[3]->Area;

					// split up the height of all of the nodes
					tConstructOct.NodePtr->Children[3]->Area.bY =
					tConstructOct.NodePtr->Children[2]->Area.bY =
					tConstructOct.NodePtr->Children[1]->Area.bY =
					tConstructOct.NodePtr->Children[0]->Area.bY = NewMiddle;

					tConstructOct.NodePtr->Children[4]->Area.tY =
					tConstructOct.NodePtr->Children[5]->Area.tY =
					tConstructOct.NodePtr->Children[6]->Area.tY =
					tConstructOct.NodePtr->Children[7]->Area.tY = NewMiddle;

					tConstructOct2.WidthSplitsLeft = tConstructOct.WidthSplitsLeft-1;
					tConstructOct2.LengthSplitsLeft = tConstructOct.LengthSplitsLeft-1;
					tConstructOct2.HeightSplitsLeft = tConstructOct.HeightSplitsLeft-1;

				
					// keep on adding them to the stack if at least one of the dimensions is still
					// splittable
					if ((tConstructOct.WidthSplitsLeft > 1 || tConstructOct.LengthSplitsLeft > 1
						|| tConstructOct.HeightSplitsLeft >  1))
					{						
						for (int ix = 0; ix < 8; ix++)
						{
							tConstructOct2.NodePtr = tConstructOct.NodePtr->Children[ix];
							ConstructStack.push(tConstructOct2);
						}						
					}
				} // end if which splits up the height
				else
				{
					// only could split up two dimensions so add those four members into 
					// the stack

					tConstructOct2.WidthSplitsLeft = tConstructOct.WidthSplitsLeft-1;
					tConstructOct2.LengthSplitsLeft = tConstructOct.LengthSplitsLeft-1;
					tConstructOct2.HeightSplitsLeft = tConstructOct.HeightSplitsLeft;

					// keep on adding them to the stack if at least one of the dimensions is still
					// splittable
					if ((tConstructOct.WidthSplitsLeft > 1 || tConstructOct.LengthSplitsLeft > 1
						|| tConstructOct.HeightSplitsLeft > 1))
					{						
						for (int ix = 0; ix < 4; ix++)
						{
							tConstructOct2.NodePtr = tConstructOct.NodePtr->Children[ix];
							ConstructStack.push(tConstructOct2);
						}						
					}
				}// end else can't split up heigh
			}// end can split up width
			else
			{
				// can't split up width but add the two length splits into the stack
				// update which dimensions are splittable			

				tConstructOct2.LengthSplitsLeft = tConstructOct.LengthSplitsLeft -1;

				tConstructOct2.WidthSplitsLeft = tConstructOct.WidthSplitsLeft;
				tConstructOct2.HeightSplitsLeft = tConstructOct.HeightSplitsLeft;

				// keep on adding them to the stack if at least one of the dimensions is still
				// splittable
				if ((tConstructOct.WidthSplitsLeft > 1 || tConstructOct.LengthSplitsLeft > 1
					|| tConstructOct.HeightSplitsLeft > 1))
				{						
					
						tConstructOct2.NodePtr = tConstructOct.NodePtr->Children[Zero];
						ConstructStack.push(tConstructOct2);
						tConstructOct2.NodePtr = tConstructOct.NodePtr->Children[One];
						ConstructStack.push(tConstructOct2);
										
				}
			}// end else can't split width
		}// end can't split length, could copy and paste to make width greater then length
		
	} // end stack is empty quit out
	CreateOctTree1(); // now join the neighbours
}

//---------------------------------------------------------------------
// RenderOctGrid(LINEVERTEX* &pVertex, int &VertexCount)
// 
// Usage: Called in the rendering loops, used to render the oct tree grid
// 
// Input: pVertex, vertex buffer ptr
// Output: VertexCount, return vertex count
// DepthLevel: The depth level of the oct tree to render
//---------------------------------------------------------------------
void OctTree::RenderOctGrid(LINEVERTEX* &pVertex, int &VertexCount, int DepthLevel)
{
	int TraversalFlag = HeadOctPtr->Flag + 1; // an arbitrary flag, so a node is only rendered once
	int CurrLevel = 1; // used to only render up to the level desired

	DWORD Color1 = 0xff000000;
	DWORD Color2 = 0xff0000ff;

	OctNodePtr Curr,Next,Goto;

	Goto = Curr = Next = HeadOctPtr;

	VertexCount = 0;


	// nothing to do, return
	if (DepthLevel == 0)
		return;

	RenderFormCube(pVertex,Goto->Area,VertexCount,Color1);

	// only top level to do, return
	if (DepthLevel == 1)
		return;

	HeadOctPtr->Flag = HeadOctPtr->Flag + 1;

	// do a walk of the oct tree visiting every node once, and adding the vertices 
	// to the vertex buffer
	// add the first one to the buffer
	do {

		Curr = Goto;
		Goto = NULL;
	
		if (CurrLevel < DepthLevel)
		{
			for (int ix = 0; ix < 8; ix++)
			{
				if (Curr->Children[ix] != NULL)
				{
					if (Curr->Children[ix]->Flag != TraversalFlag)
					{
						Goto = Curr->Children[ix];
						Goto->Flag = TraversalFlag; // don't visit this node again	
						if (Goto->ObjectList.empty())
						{
						//	RenderFormCube(pVertex,Goto->Area,VertexCount,Color1);						
						}
						else
						{
							OBJECTLIST::iterator curr = Goto->ObjectList.begin();
							OBJECTLIST::iterator end = Goto->ObjectList.end();
							while (curr != end)
							{							
								if ((*curr)->isSelected())
								{
									RenderFormCube(pVertex,Goto->Area,VertexCount,Color2);								
								}					
								curr++;
							}
						}
						// don't want to exceed the count
						if (VertexCount >= 30000)
						{
							return;
						}
						CurrLevel++; // update depth level
						break;
					}
				}
			}
		}
		if (Goto == NULL)
		{
			if (Curr->Parent == NULL)
			{
				//exit
			}
			else
			{
				Goto = Curr->Parent;
				CurrLevel--;
			}
		}
	} while (Goto != NULL);

}

//---------------------------------------------------------------------
// RenderFormCube(LINEVERTEX *&pVertex, FCube theCube, int &VertexCount)
// 
// Usage: Creates a cube by calling renderformface 6 times (one for each face)
// 
//---------------------------------------------------------------------
void OctTree::RenderFormCube(LINEVERTEX *&pVertex, const FCube &theCube, int &VertexCount, const DWORD &Color)
{
	
	D3DVECTOR Points[4];
	DWORD Color1 = Color;

	// front
	Points[0] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.tZ);
	Points[1] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.tZ);
	Points[2] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.tZ);
	Points[3] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.tZ);
	RenderFormFace(pVertex,Points,VertexCount,Color);

	// back
	Points[0] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.bZ);
	Points[1] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.bZ);
	Points[2] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.bZ);
	Points[3] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.bZ);
	RenderFormFace(pVertex,Points,VertexCount,Color);


  	pVertex[0].position = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.tZ);
	pVertex[0].color = Color1;

	pVertex[1].position = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.bZ);
	pVertex[1].color = Color1;

	pVertex[2].position = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.tZ);
	pVertex[2].color = Color1;
	
	pVertex[3].position = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.bZ);
	pVertex[3].color = Color1;

	pVertex[4].position = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.tZ);
	pVertex[4].color = Color1;

	pVertex[5].position = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.bZ);
	pVertex[5].color = Color1;	

	pVertex[6].position = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.tZ);
	pVertex[6].color = Color1;	

	pVertex[7].position = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.bZ);
	pVertex[7].color = Color1;

	VertexCount += 4;
    pVertex += 8;

	// right
/*	Points[0] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.tZ);
	Points[1] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.bZ);
	Points[2] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.bZ);
	Points[3] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.tZ);
	RenderFormFace(pVertex,Points,VertexCount);

	// left
	Points[0] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.tZ);
	Points[1] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.bZ);
	Points[2] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.bZ);
	Points[3] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.tZ);
	RenderFormFace(pVertex,Points,VertexCount);

	// top
	Points[0] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.tZ);
	Points[1] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.tZ);
	Points[2] = D3DXVECTOR3(theCube.tX,theCube.tY,theCube.bZ);
	Points[3] = D3DXVECTOR3(theCube.bX,theCube.tY,theCube.bZ);
	RenderFormFace(pVertex,Points,VertexCount);

	// bottom
	Points[0] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.tZ);
	Points[1] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.tZ);
	Points[2] = D3DXVECTOR3(theCube.tX,theCube.bY,theCube.bZ);
	Points[3] = D3DXVECTOR3(theCube.bX,theCube.bY,theCube.bZ);
	RenderFormFace(pVertex,Points,VertexCount);*/


}


//---------------------------------------------------------------------
// RenderFormFace(VERTEX *&pVertex, D3DVECTOR Points[], int &VertexCount)
// 
// Usage: Creates a face out of 4 points
// 
// Inputs:
// Point 0 Bottom left
// Point 1 Bottom right
// Point 2 Bottom top left
// Point 3 Bottom top right
// int WidthSegments: the number of width segments, must be a power of 2
// int LengthSegments: the number of length segments, must be a power of 2
// int HeightSegments: the number of height segments, must be a power of 2

// Notes: By convention the height is splittable less then the width of length
//---------------------------------------------------------------------
void OctTree::RenderFormFace(LINEVERTEX *&pVertex, const D3DVECTOR Points[], int &VertexCount, const DWORD &Color)
{
	DWORD Color1 = Color;
	// face 1
	pVertex[0].position = Points[0];
	pVertex[0].color = Color1;

	pVertex[1].position = Points[1];
	pVertex[1].color = Color1;

	pVertex[2].position = Points[1];
	pVertex[2].color = Color1;
	
	pVertex[3].position = Points[2];
	pVertex[3].color = Color1;

	pVertex[4].position = Points[2];
	pVertex[4].color = Color1;

	pVertex[5].position = Points[3];
	pVertex[5].color = Color1;	

	pVertex[6].position = Points[3];
	pVertex[6].color = Color1;	

	pVertex[7].position = Points[0];
	pVertex[7].color = Color1;

	VertexCount += 4;
    pVertex += 8;

}

//---------------------------------------------------------------------
// DeleteOctTree()
// 
// Usage: Deletes the oct tree, called when loading a new tree or 
//        in the destructer
//---------------------------------------------------------------------
void OctTree::DeleteOctTree()
{
	BOOL exit;

	if (HeadOctPtr == NULL)
	{
		return;
	}

	OctNodePtr TraversePtr,DeletePtr;

	TraversePtr = HeadOctPtr;

	exit = false;

	while (!exit)
	{
		exit = true;
		for (int ix = 0; ix < 8; ix++)
		{
			if (TraversePtr->Children[ix] != NULL)
			{
				TraversePtr = TraversePtr->Children[ix];
				exit = false;
				break;
			}
		}
		if (exit)
		{
			if (TraversePtr->Parent != NULL)
			{
				DeletePtr = TraversePtr;
				TraversePtr = TraversePtr->Parent;
				for (int ix = 0; ix < 8; ix++)
				{
					if (TraversePtr->Children[ix] == DeletePtr)
					{
						TraversePtr->Children[ix] = NULL;
						break;
					}
				}
				delete DeletePtr;				
				exit = false;
			}
		}
	}

	delete HeadOctPtr;
	HeadOctPtr = NULL;

}

//---------------------------------------------------------------------
// CreateOctTreeCN()
// 
// Usage: Called in the CreateOctTree function after all of the nodes have been
//        created to connect them to the neighbours
//
// Algorithm: For each of the six possible neighbours, it generates a point in 3-D space
//            that is contained in the neighbour and then does a search to figure out which 
//            node this point is in, this node is the neighbour. 
//---------------------------------------------------------------------
void OctTree::CreateOctTreeCN(OctNodePtr &CurrentNode, int &CurrLevel)
{
	D3DVECTOR NPoint;
	OctNodePtr tOctNodePtr;

	int tCurrLevel;

	// must have at least one parent
	if (CurrentNode->Parent == NULL)
		return; 

	if ((CurrentNode->Area.bX == 0) &&
		(CurrentNode->Area.bY == 0) &&
		(CurrentNode->Area.bZ == 0))
	{
		int asdf = 34;
		asdf = 23;
	}

	for (int ix = 0; ix < 6; ix++)
	{
		ASSERT(CurrentNode != NULL);

		NPoint = CreateOctTreeGPoint(CurrentNode->Area,ix);	

		// the point might not be valid so in that case don't bother checking
		if (HeadOctPtr->Area.PointinCube(NPoint))
		{
		
			tOctNodePtr = CurrentNode->Parent;
			tCurrLevel = CurrLevel-1;
			// go up the tree to find a parent that contains this point
			while (tOctNodePtr != NULL)
			{
				if (tOctNodePtr->Area.PointinCube(NPoint))
				{				
					break;
				}
				tCurrLevel--;
				tOctNodePtr = tOctNodePtr->Parent;
			}

			// now go downwards to the children to find one in the same level that matches,
			// note at this point there must be a match
			if (tOctNodePtr != NULL)
			{
				while (tCurrLevel != CurrLevel)
				{
					OctNodePtr tempo; // debug
					ASSERT(tOctNodePtr != NULL);

					int newIndex = UGetChildIndex(tOctNodePtr->Area,NPoint,tOctNodePtr);				

					tempo = tOctNodePtr;// debug

					tOctNodePtr = tOctNodePtr->Children[newIndex];
					tCurrLevel++;
				}
				CurrentNode->Neighbours[ix] = tOctNodePtr;
			}
		}
	}

}

//---------------------------------------------------------------------
// CreateOctTreeGPoint(FCube CurrBounds, int DesiredN)
// 
// Usage: To Get a point that is contained in the neighbours
//
// Inputs: DesiredN the index of the neighbour from 0-5.
//		   CurrBounds the current bounds of the node
// Outputs: D3DVECTOR, a point that is contained in the neighbours cube
//---------------------------------------------------------------------
inline D3DVECTOR OctTree::CreateOctTreeGPoint(FCube &CurrBounds, int &DesiredN)
{
	D3DVECTOR tReturn;
	float SmallNumber = 0.1f;
	if (DesiredN == 0)
	{
		// +Z neighbour
		tReturn.x = CurrBounds.tX - CurrBounds.Length()/2.0f;
		tReturn.y = CurrBounds.tY - CurrBounds.Height()/2.0f;
		tReturn.z = CurrBounds.tZ + SmallNumber;
	}
	else if (DesiredN == 1)
	{
		// +X neighbour
		tReturn.z = CurrBounds.tZ - CurrBounds.Width()/2.0f;
		tReturn.y = CurrBounds.tY - CurrBounds.Height()/2.0f;
		tReturn.x = CurrBounds.tX + SmallNumber;
	}
	else if (DesiredN == 2)
	{
		// -Z neighbour
		tReturn.x = CurrBounds.tX - CurrBounds.Length()/2.0f;
		tReturn.y = CurrBounds.tY - CurrBounds.Height()/2.0f;
		tReturn.z = CurrBounds.bZ - SmallNumber;
	}
	else if (DesiredN == 3)
	{
		// -X neighbour
		tReturn.z = CurrBounds.tZ - CurrBounds.Width()/2.0f;
		tReturn.y = CurrBounds.tY - CurrBounds.Height()/2.0f;
		tReturn.x = CurrBounds.bX - SmallNumber;
	}
	else if (DesiredN == 4)
	{
		// +Y neighbour
		tReturn.z = CurrBounds.tZ - CurrBounds.Width()/2.0f;
		tReturn.x = CurrBounds.tX - CurrBounds.Length()/2.0f;
		tReturn.y = CurrBounds.tY + SmallNumber;
	}
	else if (DesiredN == 5)
	{
		// -Y neighbour
		tReturn.z = CurrBounds.tZ - CurrBounds.Width()/2.0f;
		tReturn.x = CurrBounds.tX - CurrBounds.Length()/2.0f;
		tReturn.y = CurrBounds.bY - SmallNumber;
	}

	return tReturn;
}

//---------------------------------------------------------------------
// UGetChildIndex(FCube &Cube, D3DVECTOR &Point)
// 
// Usage: Gets the child index from a parent based on a point that
//	      must be contained in the parent, there is no error checking otherwise

// Inputs:
// Cube, the cube enclosed by the parent
// Point:  The point contains in the parent and only one of the children
//
// Returns the index of the child, or -1 if error
//
// Note: 1) Does not use FCube::PointinCube for each child because these calls 
//          are more expensive
//		 2) Due to the unbalanced nature of the octree NULL checks are required throughout
//		
//---------------------------------------------------------------------
inline int OctTree::UGetChildIndex(FCube &Cube, D3DVECTOR &Point, OctNodePtr &CurrNode)
{
	if (Point.x >= (Cube.tX - Cube.Length()/2.0f))
	{
		if (Point.y >= (Cube.tY - Cube.Height()/2.0f))
		{
			if (Point.z <= (Cube.tZ - Cube.Width()/2.0f))
			{
				return 0;
			}
			else
			{
				// in an unbalanced octtree this is a possible case, should return lower 
				// order one in this case
				if (CurrNode->Children[3] == NULL)
					return 0;				

				return 3;
			}
		}
		else
		{
			if (Point.z <= (Cube.tZ - Cube.Width()/2.0f))
			{
				if (CurrNode->Children[4] == NULL)
					return 0;

				return 4;
			}
			else
			{
				if (CurrNode->Children[7] == NULL)
					if (CurrNode->Children[3] == NULL)
						return 0;
					else
						return 3;

				return 7;
			}
		}
	}
	else
	{
		if (Point.y >= (Cube.tY - Cube.Height()/2.0f))
		{
			if (Point.z <= (Cube.tZ - Cube.Width()/2.0f))
			{
				if (CurrNode->Children[1] == NULL)
					return 0;
				else
					return 1;
			}
			else
			{
				// in an unbalanced octtree this is a possible case, should return lower 
				// order one in this case
				if (CurrNode->Children[2] == NULL)
					if(CurrNode->Children[1] == NULL)
						return 3;
					else
						return 1;

				return 2;
			}
		}
		else
		{
			if (Point.z <= (Cube.tZ - Cube.Width()/2.0f))
			{				
				if (CurrNode->Children[5] == NULL)
					if (CurrNode->Children[1] == NULL)
						return 0;
					else
						return 1;

				return 5;
			}
			else
			{
				if (CurrNode->Children[6] == NULL)
					if (CurrNode->Children[2] == NULL)
						if (CurrNode->Children[1] == NULL)
							return 3;
						else
							return 1;
					else
						return 2;

				return 6;
			}
		}
	}

	return -1;
}

//---------------------------------------------------------------------
// CreateOctTree1()
// 
// Usage: Called after CreateOctTree() where all of the octnodes are created
//		  to join the neighbours
//
// Algorithm: Traverse the tree visiting each node once, while joining the node
//            to its neighbours
//---------------------------------------------------------------------
void OctTree::CreateOctTree1()
{
	int TraversalFlag = HeadOctPtr->Flag + 1; // an arbitrary flag, so a node is only rendered once
	int CurrLevel = 1; 

	OctNodePtr Curr,Next,Goto;

	Goto = Curr = Next = HeadOctPtr;

	HeadOctPtr->Flag = HeadOctPtr->Flag + 1;

	// do a walk of the oct tree visiting every node once, and adding the vertices 
	// to the vertex buffer
	// add the first one to the buffer
	do {

		Curr = Goto;
		Goto = NULL;
	
	
		for (int ix = 0; ix < 8; ix++)
		{
			if (Curr->Children[ix] != NULL)
			{
				if (Curr->Children[ix]->Flag != TraversalFlag)
				{
					Goto = Curr->Children[ix];
					Goto->Flag = TraversalFlag; // don't visit this node again					
					CreateOctTreeCN(Goto,CurrLevel);					
					CurrLevel++; // update depth level
					break;
				}
			}
		}
	
		if (Goto == NULL)
		{
			if (Curr->Parent == NULL)
			{
				//exit
			}
			else
			{
				Goto = Curr->Parent;
				CurrLevel--;
			}
		}
	} while (Goto != NULL);


}

//---------------------------------------------------------------------
// AddStartSequence()
// 
// Usage: Called when starting to add an object to the oct tree
//
// Why?:  Because an object is added to the octree by calling a function
//        which only reference a leaf node.  The function then traverse upwards
//        to its parents. Of course it in uneccessary to travel to a parent node twice
//        so an interator keeps track if a node was already visited
//---------------------------------------------------------------------
void OctTree::AddStartSequence()
{
	AddIterator = HeadOctPtr->Flag + 1;
}


//---------------------------------------------------------------------
// AddObjecttoNode(StdObjectPtr theObject, OctNodePtr theNode)
// 
// Usage: Adds a object to the node, plus all of the parents nodes
//
// Algorithm: Traverse up the tree and adds the object to the node
//---------------------------------------------------------------------
void OctTree::AddObjecttoNode(StdObjectPtr theObject, OctNodePtr theNode)
{
	BOOL Done = false;

	while (!Done)
	{
		// check if it should be added
		if (theNode->Flag != AddIterator)
		{
			theNode->ObjectList.push_back(theObject);
			theNode->Flag = AddIterator;
		}
		else
			Done = true;
		// traverse upwards
		if (theNode->Parent != NULL)
		{
			theNode = theNode->Parent;
		}
		else
			Done = true;
	}

}

//---------------------------------------------------------------------
// RenderAllObjects(LPDIRECT3DDEVICE8 m_pd3dDevice)
// 
// Usage: Temp hack to render all of the objects
//
//---------------------------------------------------------------------
void OctTree::RenderAllObjects(LPDIRECT3DDEVICE8 m_pd3dDevice)
{
	OBJECTLIST::iterator curr = HeadOctPtr->ObjectList.begin();
	OBJECTLIST::iterator end = HeadOctPtr->ObjectList.end();
	while (curr != end)
	{
		// add the object to the octnode
		(*curr)->Render(m_pd3dDevice);
		curr++;
	} 
}

//---------------------------------------------------------------------
// UGetPlane(OctNodePtr tOctNodePtr, int neighbour)
// 
// Usage: Helper function, returns a plane structure the nieghbour bounds
//        of a oct cube
//
//---------------------------------------------------------------------
PLANE OctTree::UGetPlane(OctNodePtr tOctNodePtr, int neighbour)
{
	PLANE tReturn;
	if (neighbour == 0)
	{
		// + Z
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
	}
	else if (neighbour == 1)
	{
		// + X
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
	}
	else if (neighbour == 2)
	{
		// - Z
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
	}
	else if (neighbour == 3)
	{
		// - X
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
	}
	else if (neighbour == 4)
	{
		// + Y
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.tZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.tY,tOctNodePtr->Area.bZ);
	}
	else if (neighbour == 5)
	{
		// - Y
		tReturn.v1 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
		tReturn.v2 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.tZ);
		tReturn.v3 = D3DXVECTOR3(tOctNodePtr->Area.bX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
		tReturn.v4 = D3DXVECTOR3(tOctNodePtr->Area.tX,tOctNodePtr->Area.bY,tOctNodePtr->Area.bZ);
	}

	return tReturn;
}

//---------------------------------------------------------------------
// SelectObject(const Ray &ray)
// 
// Usage: When the user has clicked on the screen to select an object, this function
//        is eventually called
//
// Algorithm: 1) Two cases, either the origin is inside the oct tree or not
//            2) If inside find the octant which it is in to start the search
//            3) If it isn't find where the ray intersects the main octree cube to start the intersection test
//            4) Search every octant in the path of ray until octree exited or object found
//
//			  5) Perform any neccessary operation to add to select group
//---------------------------------------------------------------------
void OctTree::SelectObject(const RAY &Ray, BOOL AddtoGroup)
{

	OctNodePtr StartNode;
	BOOL DisableOptimzation = true; // the travel only in relavant octants doesn't quite work yet
	
	// 1) Find if the origin is in the octree or not
	if (HeadOctPtr->Area.PointinCube(D3DVECTOR(Ray.origin)))
	{
		// 2) O.K now search where the origin is located in.
		StartNode = HeadOctPtr;
		if (!DisableOptimzation)
		{
			while (StartNode->Children[0] != NULL)
			{
				int Index = UGetChildIndex(StartNode->Area,D3DVECTOR(Ray.origin),StartNode);
				StartNode = StartNode->Children[Index];
			}
		}
	}
	else
	{
		// 3) O.K now search where the origin is located in.
		float Distance = 9999999.9f;
		float tDistance;
		D3DXVECTOR3 IntersectPoint,tIntersectPoint;
		PLANE tPlane;
		// O.K find the intersection point on the octree
		for (int ix = 0; ix < 6; ix++)
		{
			tPlane = UGetPlane(HeadOctPtr,ix);
			if (Math.IntersectPlane(tPlane,Ray,tIntersectPoint))
			{
				// find the intersection point.
				tDistance = Math.GetDistance(Ray.origin,tIntersectPoint);
				if (tDistance < Distance)
				{
					Distance = tDistance;
					IntersectPoint = tIntersectPoint;
				}
			}
		}
		// O.K now search where the origin is located in.
		StartNode = HeadOctPtr;
		if (!DisableOptimzation)
		{
			while (StartNode->Children[0] != NULL)
			{
				int Index = UGetChildIndex(StartNode->Area,D3DVECTOR(IntersectPoint),StartNode);
				StartNode = StartNode->Children[Index];
			}
		}

	}
	
	float Distance = 999999.9f;
	StdObjectPtr SelectedObject = NULL;

	// 4) Have starting octant, now proceed checking only neighbouring octants
	D3DXVECTOR3 Intersection;

	// first check the starting octant
	OBJECTLIST::iterator curr = StartNode->ObjectList.begin();
	OBJECTLIST::iterator end = StartNode->ObjectList.end();
	while (curr != end)
	{
		// add the object to the octnode
		if ((*curr)->IntersectBoundingSphere(Ray,Intersection))
		{
			if ((*curr)->IntersectBoundingBox(Ray,Intersection))
			{
				float tDistance = Math.GetDistance(Ray.origin,Intersection);
				if (tDistance < Distance)
				{
					Distance = tDistance;
					SelectedObject = *curr;
				}
			}
		}
				
		curr++;
	} 	


	// now check other octants (the neighbours)

	BOOL Done;
	float TraverseIterator = HeadOctPtr->Flag + 1;
	PLANE tPlane;

	if (SelectedObject != NULL)
	{
		Done = true;
	}
	else
	{
		 // set the flag
		Done = false;
		StartNode->Flag = TraverseIterator; // because of the first check
	}

	while (!Done)
	{
		Done = true;
		for (int ix = 0; ix < 6; ix++)
		{
			if (StartNode->Neighbours[ix] != NULL)
			{
				if (StartNode->Neighbours[ix]->Flag != TraverseIterator)
				{
					// check intersection with ray
					tPlane = UGetPlane(StartNode,ix);

					if (Math.IntersectPlane(tPlane,Ray,Intersection))
					{ 					
						StartNode = StartNode->Neighbours[ix];
						StartNode->Flag = TraverseIterator;
						Done = false;

						// this is the neigbour check for objects 
						OBJECTLIST::iterator curr = StartNode->ObjectList.begin();
						OBJECTLIST::iterator end = StartNode->ObjectList.end();
						while (curr != end)
						{
							// add the object to the octnode
							if ((*curr)->IntersectBoundingSphere(Ray,Intersection))
							{
								if ((*curr)->IntersectBoundingBox(Ray,Intersection))
								{
									float tDistance = Math.GetDistance(Ray.origin,Intersection);
									if (tDistance < Distance)
									{
										Distance = tDistance;
										SelectedObject = *curr;
									}									
								}
							}
							curr++;
						} 
						if (SelectedObject != NULL)
						{
							Done = true;
						}
					}
				}
			}
		}
	}
	ASSERT(TraverseIterator > 0);
	HeadOctPtr->Flag = TraverseIterator;


	// now add to group list
	if (!AddtoGroup)
	{
		// have to nuke previous grouping
		if (!SelectList.empty())
		{
			// Deselect the currently selected objects
			GROUPLIST::iterator curr = SelectList.begin();
			GROUPLIST::iterator end = SelectList.end();
			while (curr != end)
			{
				(*curr)->OnSelect(false); // call this so the entire group hierachy get deselected
				curr++;
			}
			SelectList.clear();
		}	
	}

	if (SelectedObject != NULL)
	{
		// check to see if it is already in the select list, don't want to add it twice
		BOOL Found = false;
		GROUPLIST::iterator curr = SelectList.begin();
		GROUPLIST::iterator end = SelectList.end();
		while (curr != end)
		{
			if ((*curr) == SelectedObject->GroupPtr)
				Found = true;
			curr++;
		}
		if (!Found)
		{
			GroupStruct *TopLevel = SelectedObject->GroupPtr->OnSelect(true); // call this so the entire group hierachy get selected
			SelectList.push_back(TopLevel);
		}
	}
}

//---------------------------------------------------------------------
// SelectCreateGroup()
// 
// Usage: When the user desides so group together selected objects
//
// Algorithm: If there are more than one group pointer in the SelectList, it makes a group out of them
//---------------------------------------------------------------------
void OctTree::SelectCreateGroup()
{
	if (SelectList.size() > 1)
	{
		// O.K group em together.
		GroupStruct* GroupPtr = new GroupStruct;
		GroupStruct* Prev;

		GroupPtr->GroupType = GroupSeperable; // make it Seperable

		GROUPLIST::iterator curr = SelectList.begin();
		GROUPLIST::iterator end = SelectList.end();

		Prev = GroupPtr->Child = (*curr);
		(*curr)->Parent = GroupPtr;
		GroupPtr->NumofObjects = (*curr)->NumofObjects;
		GroupPtr->HierarchyLevel = (*curr)->HierarchyLevel+1;
		curr++;

		while (curr != end)
		{
			Prev->Sibling = (*curr);
			Prev = (*curr);
			(*curr)->Parent = GroupPtr;
			GroupPtr->NumofObjects += (*curr)->NumofObjects;

			if ((*curr)->HierarchyLevel >= GroupPtr->HierarchyLevel)
				GroupPtr->HierarchyLevel = (*curr)->HierarchyLevel+1;		

			curr++;			
		}
		SelectList.clear();
		SelectList.push_back(GroupPtr);
	}

}

//---------------------------------------------------------------------
// SelectDeleteGroup()
// 
// Usage: When the user desides so group together selected objects
//
// Algorithm: If there are more than one group pointer in the SelectList, it makes a group out of them
//---------------------------------------------------------------------
void OctTree::SelectDeleteGroup()
{
	if (SelectList.size() == 1)
	{		
		GROUPLIST::iterator group = SelectList.begin();		
		if ((*group)->GroupType == GroupSeperable)
		{
			(*group)->OnSelect(false);
			GroupStruct *GroupPtr = (*group);
			SelectList.clear();
			
			// O.K kill this level of group heiarchy
			ASSERT (GroupPtr->Child != NULL);
			if (GroupPtr->Child == NULL)
				return;

			GroupStruct* Curr = GroupPtr->Child;
			GroupStruct* Prev = NULL;

			while (Curr != NULL)
			{
				Curr->Parent = NULL;
				Prev = Curr;				
				Curr = Curr->Sibling;
				Prev->Sibling = NULL;
				SelectList.push_front(Prev);				
				Prev->OnSelect(true);
			}
			delete GroupPtr;			
			
		}
	}
}

//=====================================================================
// RemoveObjectfromNode(StdObject *theObject, OctNode *OctNodePtr)
// 
// Usage: Removes an object from an OctNode if it is there
//
// Algorithm: If there are more than one group pointer in the SelectList, it makes a group out of them
//=====================================================================
void OctTree::RemoveObjectfromNode(StdObject *theObject, OctNode *theNode)
{
	BOOL Done = false;

	while (!Done)
	{
		// check if it should be added
		if (theNode->Flag != AddIterator)
		{
			theNode->ObjectList.remove(theObject);
			theNode->Flag = AddIterator;
		}
		else
			Done = true;
		// traverse upwards
		if (theNode->Parent != NULL)
		{
			theNode = theNode->Parent;
		}
		else
			Done = true;
	}

}

//=====================================================================
// ClearOctTree()
// 
// Usage: Nukes all the contents of the octree but keeps the structure
//
//=====================================================================
void OctTree::ClearOctTree()
{
	int TraversalFlag = HeadOctPtr->Flag + 1; // an arbitrary flag, so a node is only rendered once

	OctNodePtr Curr,Next,Goto;

	Goto = Curr = Next = HeadOctPtr;

	HeadOctPtr->Flag = HeadOctPtr->Flag + 1;

	// do a walk of the oct tree visiting every node once, and clearing the object list as we go along
	do {

		Curr = Goto;
		Goto = NULL;
	
		for (int ix = 0; ix < 8; ix++)
		{
			if (Curr->Children[ix] != NULL)
			{
				if (Curr->Children[ix]->Flag != TraversalFlag)
				{
					Goto = Curr->Children[ix];
					Goto->Flag = TraversalFlag; // don't visit this node again	
					Curr->Children[ix]->ObjectList.clear();	
					break;
				}
			}			
		}
		if (Goto == NULL)
		{
			if (Curr->Parent == NULL)
			{
				//exit
			}
			else
			{
				Goto = Curr->Parent;
			}
		}
	} while (Goto != NULL);

	HeadOctPtr->ObjectList.clear();
}
