// TileEditView.cpp: implementation of the TileEditView class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "MapEditor.h"
#include "TileEditView.h"
#include "MessageMap.h"
#include "ChildView.h"
#include "TerrainManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

TileEditView::TileEditView()
{

	RollOutManager.ScrollViewPtr = this;
}

//=====================================================================
// TileEditView(CRect WindowSize, CWnd *ParentPtr)
// 
// Usage: "New" Constructer
//
//=====================================================================
TileEditView::TileEditView(CRect WindowSize, CWnd *ParentPtr)
{
	Create(WindowSize,ParentPtr);
}

//=====================================================================
// TileEditView(CRect WindowSize, CWnd *ParentPtr)
// 
// Usage: Main creation function
//
//=====================================================================
void TileEditView::Create(CRect WindowSize, CWnd *ParentPtr)
{
	// do a simply assignments
	ASSERT (EditObjectPtr != NULL);
	theTilePtr = (TileObject *)EditObjectPtr;

	for (int ix = 0; ix < MAXCALLBACK; ix++)
		CallbackFunctions[ix] = NULL;	

	// doesn't matter because this is dynamically adjusted by the view
	CPoint ViewSize;
	ViewSize.x = 200; // make sure this is 200 though
	ViewSize.y = 500;
	ScrollView::Create(WindowSize,ViewSize,ParentPtr);

	AddBasicEditing();
	AddGaussianEditing();
	AddSinusoidalEditing();
	ReDrawBars();
	
	// debugging
	RollOutManager.AddGroup("Jello Editing",true);
	ResetVerticalSize(RollOutManager.GetCurrAddYOffset());

	theTerrainManager.OnSelectNewTile(); // increment flag
}

// Destructer
TileEditView::~TileEditView()
{
	theTerrainManager.ResetTerrainInOctree(); // just in case the user forgot

}

BEGIN_MESSAGE_MAP(TileEditView, ScrollView)
	//{{AFX_MSG_MAP(ScrollView)
	ON_MESSAGE(WM_ONUPDATESPINEDIT,OnSpinEditControlMessage)
	ON_WM_LBUTTONUP()
	ON_BN_CLICKED(GAUSSIANENABLE,On_G2Enable)
	ON_BN_CLICKED(SINUSOIDALENABLE,On_G3Enable)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()


//=====================================================================
// Paint(CDC *pDC)
// 
// Usage: Virtual overridden so the scroll view could indicate a redraw is happening
//		  In this case it tells the rolloutmanager to repaint
//
//=====================================================================
void TileEditView::Paint(CDC *pDC)
{
	RollOutManager.OnPaint(pDC);
}




//=====================================================================
// OnSpinEditControlMessage(WPARAM wParam, LPARAM lParam)
// 
// Usage: Messages handler
//
//=====================================================================
LRESULT TileEditView::OnSpinEditControlMessage(WPARAM wParam, LPARAM lParam)
{
	int ControlID = (int &)wParam;
	if (CallbackFunctions[ControlID] != NULL)	
	{
		(*this.*CallbackFunctions[ControlID])();
	}	

	return 0;
}

//=====================================================================
// OnTileHeight()
// 
// Usage: Messages handler Callback function, for TileHeight spineditcontrol
//
//=====================================================================
void TileEditView::On_G1TileHeight()
{
	float DeltaHeight = g1_TileHeight.GetValue() - PrevHeight;
	theTilePtr->SetHeightD(DeltaHeight); // use the delta height
	PrevHeight = g1_TileHeight.GetValue();

	// Get the new values for each of the 4 vertices controls
	float tHeight[4];
	theTilePtr->GetTileHeight(tHeight);
	for (int ix = 0; ix < 4; ix++)
		g1_Vertices[ix].SetValue(tHeight[ix]);

	// redraw
	theTerrainManager.InvalidateTerrainRender();
	theViewPtr->Redraw();

}

//=====================================================================
// OnTileHeight()
// 
// Usage: Messages handler Callback function, for all four vertices spineditcontrols
//
//=====================================================================
void TileEditView::On_G1VerticesChange()
{
	float Height[4];
	float MaxHeight = 0.0f;
	for (int ix = 0; ix < 4; ix++)
	{
		Height[ix] = g1_Vertices[ix].GetValue();
		if (Height[ix] > MaxHeight)
			MaxHeight = Height[ix];
	}
	// make the change
	theTilePtr->SetHeight(Height);
	// ajdust the value of the tileheight control
	PrevHeight = MaxHeight;
	g1_TileHeight.SetValue(MaxHeight);

	// redraw
	theTerrainManager.InvalidateTerrainRender();
	theViewPtr->Redraw();
}

//=====================================================================
// AddBasicEditing()
// 
// Usage: Adds the basic editing group to the roll out manager
//
//=====================================================================
void TileEditView::AddBasicEditing()
{

	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();	

	// Create the group
	RollOutManager.AddGroup("Basic Editing",true);

	// get the offset to start adding the groups controls
	int YOffset = RollOutManager.GetCurrAddYOffset();

	// using right justification
	CPoint CreationOffset;
	CreationOffset.x = 170; 
	CreationOffset.y = YOffset;

	g1_TileHeight.Create("TileHeight:",CreationOffset,0,Info.Height,50,SE_FLOAT | SE_RIGHTJUSTIFIED,this,TILEHEIGHTCONTROL);
	// adds its callback function
	CallbackFunctions[TILEHEIGHTCONTROL] = On_G1TileHeight;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g1_TileHeight,g1_TileHeight.GetWindowRect());
		
	for (int ix = 0; ix < 4; ix++)
	{
		CreationOffset.y += 25;
	
		char tempo[30];
		sprintf(tempo,"Vertex %d:",ix+1);
		g1_Vertices[ix].Create(tempo,CreationOffset,0,Info.Height,50,SE_FLOAT|SE_RIGHTJUSTIFIED/*SE_STDSPACEJUSTIFIED*/,this,VERTICESCONTROL+ix);
		// adds its callback function
		CallbackFunctions[VERTICESCONTROL+ix] = On_G1VerticesChange;
		// add it to the roll out
		RollOutManager.AddControltoGroup((CWnd *)&g1_Vertices[ix],g1_Vertices[ix].GetWindowRect());
	}

	// set the initial values of the controls by getting the correct values
	// from the tile
	float tHeight[4];
	theTilePtr->GetTileHeight(tHeight);
	float Max = 0;
	for (ix = 0; ix < 4; ix++)
	{
		if (tHeight[ix] > Max)
			Max = tHeight[ix];
		g1_Vertices[ix].SetValue(tHeight[ix]);
	}
	PrevHeight = Max;
	g1_TileHeight.SetValue(Max);


}

//=====================================================================
// OnLButtonUp(UINT nFlags, CPoint point) 
// 
// Usage: Doesn't do much other then send the message along to the roll out manager
//        which is a generic class, (not a window), however, it excepts a return
//        value which determines whether to redraw the window
//
//=====================================================================
void TileEditView::OnLButtonUp(UINT nFlags, CPoint point) 
{
	CPoint tPoint = this->GetCurrentScrollOffset();
	point.x += tPoint.x;
	point.y += tPoint.y;
	CPoint InvalidateOffset = RollOutManager.OnLButtonUp(nFlags,point);
	if (InvalidateOffset.y != -1)
	{
		// for an invalidate rect
		CRect tInvalidateRect;
		CRect ViewSize = GetViewSize();
		CPoint ScrollOffset = this->GetCurrentScrollOffset();
		tInvalidateRect.left = 0 - ScrollOffset.x;
		tInvalidateRect.top = InvalidateOffset.y - ScrollOffset.y;
		tInvalidateRect.right = ViewSize.right - ScrollOffset.x;
		tInvalidateRect.bottom = ViewSize.bottom - ScrollOffset.y;
		InvalidateRect(tInvalidateRect,false);
		UpdateWindow();
		ReDrawBars();
	}

	
	CWnd ::OnLButtonUp(nFlags, point);
}


//=====================================================================
// OnResetSize()
// 
// Usage: Virtual function call overridden to help the scroll manager dynamically
//		  resize the scroll view.  It calls the Roll out manager which adjusts
//        the control windows contained in it.
//=====================================================================
void TileEditView::OnResetSize()
{
	RollOutManager.ResetPositions();

}

//=====================================================================
// AddGaussianEditing()
// 
// Usage: Adds the gaussian editing group
//=====================================================================
void TileEditView::AddGaussianEditing()
{
	// Create the group
	RollOutManager.AddGroup("Gaussian Tool",true);

	// get the offset to start adding the groups controls
	int YOffset = RollOutManager.GetCurrAddYOffset();

	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();	

	// using right justification
	CPoint CreationOffset;
	CreationOffset.x = 180; 
	CreationOffset.y = YOffset;
	
	// create the tool enable toggle button
	CRect CreationRect;
	CreationRect.top = YOffset;
	CreationRect.bottom = YOffset+12;
	CreationRect.left = 120;
	CreationRect.right = 180;
	g2_Enable.Create("Enable",BS_AUTOCHECKBOX | WS_CHILD|WS_VISIBLE,CreationRect,this,GAUSSIANENABLE);
	g2_Enable.SetFont(Fonts.GetFont(MSSANSSERIF),false);
	g2_Enable.SetCheck(0);
	CallbackFunctions[GAUSSIANENABLE] = On_G2Enable;
	g2_isEnabled = false;
	RollOutManager.AddControltoGroup((CWnd *)&g2_Enable,CreationRect);
	
	CreationOffset.y = YOffset += 20;

	float tHeight[4];
	theTilePtr->GetTileHeight(tHeight);
	float Max = 0;
	for (int ix = 0; ix < 4; ix++)
	{
		if (tHeight[ix] > Max)
			Max = tHeight[ix];
	}

	g2_HeightConstant.Create("Height Constant:",CreationOffset,0,Info.Height,50,SE_FLOAT | SE_RIGHTJUSTIFIED,this,GAUSSIANHEIGHTCONSTANT);
	g2_HeightConstant.SetValue(Max);
	// adds its callback function
	CallbackFunctions[GAUSSIANHEIGHTCONSTANT] = On_G2HeightConstant;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g2_HeightConstant,g2_HeightConstant.GetWindowRect());

	CreationOffset.y = YOffset += 25;
	
	// create the sigma value spin edit control
	g2_Sigma.Create("Sigma:",CreationOffset,0,20,200,SE_FLOAT | SE_RIGHTJUSTIFIED,this,GAUSSIANSIGMA);
	g2_Sigma.SetValue(3.0f);
	// adds its callback function
	CallbackFunctions[GAUSSIANSIGMA] = On_G2Sigma;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g2_Sigma,g2_Sigma.GetWindowRect());	
}

//=====================================================================
// On_G2HeightConstant()
// 
// Usage: Message handler, for G2HeightConstant spin edit control
//=====================================================================
void TileEditView::On_G2HeightConstant()
{
	if (g2_isEnabled)
	{
		float Height = g2_HeightConstant.GetValue();
		float Sigma = g2_Sigma.GetValue();
		float Args[2];
		Args[0] = Sigma;
		Args[1] = Height;

		theTerrainManager.ExecuteFunction(theTilePtr,Args,&Gaussian);
		// redraw
		theViewPtr->Redraw();
	}

}

//=====================================================================
// On_G2Sigma()
// 
// Usage: Message handler, for G2Sigma spin edit control
//=====================================================================
void TileEditView::On_G2Sigma()
{
	if (g2_isEnabled)
	{
		float Height = g2_HeightConstant.GetValue();
		float Sigma = g2_Sigma.GetValue();
		float Args[2];
		Args[0] = Sigma;
		Args[1] = Height;

		theTerrainManager.ExecuteFunction(theTilePtr,Args,&Gaussian);
		// redraw
		theViewPtr->Redraw();
	}

}

//=====================================================================
// On_G2Enable()
// 
// Usage: Message handler, enabled or disables the guassian tool
//=====================================================================
void TileEditView::On_G2Enable()
{
	g2_isEnabled = g2_Enable.GetCheck();
}

//=====================================================================
// On_G3Enable()
// 
// Usage: Message handler, enabled or disables the sinusiodal tool
//=====================================================================
void TileEditView::On_G3Enable()
{
	g3_isEnabled = g3_Enable.GetCheck();
}

//=====================================================================
// On_G3Arg0()
// 
// Usage: Message handler, for G3HeightConstant spin edit control
//=====================================================================
void TileEditView::On_G3HeightConstant()
{
	if (g3_isEnabled)
	{
		float Args[3];
		// first fill up the args (common to all)
		Args[0] = g3_Period.GetValue();
		Args[1] = g3_HeightConstant.GetValue();
		Args[2] = g3_LDRadius.GetValue(); // may not be used depending on modes
		// first get the function from the select combo box
		int Index1 = g3_SelectMode1.GetCurSel();
		// now get the mode from the select combo box
		int Index2 = g3_SelectMode2.GetCurSel();

		if (Index2 == 0)
			theTerrainManager.ExecuteFunction(theTilePtr,Args,(TerrainFunctions[Index1][Index2]));
		else if (Index2 == 3)
			theTerrainManager.ExecuteFunctionB(theTilePtr,Args,(TerrainFunctions[Index1][Index2-3]));
		else
			theTerrainManager.ExecuteFunctionA(theTilePtr,Args,(TerrainFunctions[Index1][Index2]));

		theViewPtr->Redraw();
	}

}

//=====================================================================
// On_G3Arg1()
// 
// Usage: Message handler, for G3Periodspin edit control
//=====================================================================
void TileEditView::On_G3Period()
{
	if (g3_isEnabled)
	{
		float Args[3];
		// first fill up the args (common to all)
		Args[0] = g3_Period.GetValue();
		Args[1] = g3_HeightConstant.GetValue();
		Args[2] = g3_LDRadius.GetValue(); // may not be used depending on modes
		// first get the function from the select combo box
		int Index1 = g3_SelectMode1.GetCurSel();
		// now get the mode from the select combo box
		int Index2 = g3_SelectMode2.GetCurSel();
		
		if (Index2 == 0)
			theTerrainManager.ExecuteFunction(theTilePtr,Args,(TerrainFunctions[Index1][Index2]));		
		else if (Index2 == 3)
			theTerrainManager.ExecuteFunctionB(theTilePtr,Args,(TerrainFunctions[Index1][Index2-3]));
		else
			theTerrainManager.ExecuteFunctionA(theTilePtr,Args,(TerrainFunctions[Index1][Index2]));

		theViewPtr->Redraw();
	}
}

//=====================================================================
// On_G3LDRadius()
// 
// Usage: Message handler, for G3LDRadius edit control
//=====================================================================
void TileEditView::On_G3LDRadius()
{
	if (g3_isEnabled)
	{
		float Args[3];
		// first fill up the args (common to all)
		Args[0] = g3_Period.GetValue();
		Args[1] = g3_HeightConstant.GetValue();
		Args[2] = g3_LDRadius.GetValue(); // may not be used depending on modes
		// first get the function from the select combo box
		int Index1 = g3_SelectMode1.GetCurSel();
		// now get the mode from the select combo box
		int Index2 = g3_SelectMode2.GetCurSel();
		// adjust period for modes other then the hill mode


		if (Index2 == 0 || Index2 == 3)
			return; // not valid spin edit control for a hill option
		else
			theTerrainManager.ExecuteFunctionA(theTilePtr,Args,(TerrainFunctions[Index1][Index2]));

		theViewPtr->Redraw();
	}
}

//=====================================================================
// AddSinusoidalEditing()
// 
// Usage: Adds the sinusiodal editing group
//=====================================================================
void TileEditView::AddSinusoidalEditing()
{

	// Create the group
	RollOutManager.AddGroup("Sinusiodal Tool",true);

	// get the offset to start adding the groups controls
	int YOffset = RollOutManager.GetCurrAddYOffset();

	OctTree::OctTreeInfo Info = theOctTree.GetOctTreeInfo();	

	// using right justification
	CPoint CreationOffset;
	CreationOffset.x = 180; 
	CreationOffset.y = YOffset;
	
	// create the tool enable toggle button
	int Length = 18 + 6*6; // 18 for the button, 6 for each letter	
	CRect CreationRect;
	CreationRect.top = YOffset;
	CreationRect.bottom = YOffset+12;
	CreationRect.left = 120;
	CreationRect.right = CreationRect.left + Length;
	g3_Enable.Create("Enable",BS_AUTOCHECKBOX | WS_CHILD|WS_VISIBLE,CreationRect,this,SINUSOIDALENABLE);
	g3_Enable.SetFont(Fonts.GetFont(MSSANSSERIF),false);
	g3_Enable.SetCheck(0);
	CallbackFunctions[SINUSOIDALENABLE] = On_G3Enable;
	g3_isEnabled = false;
	RollOutManager.AddControltoGroup((CWnd *)&g3_Enable,CreationRect);

	CreationRect.left = 20;
	CreationRect.right = 180;
	CreationRect.top = CreationRect.bottom + 5;
	CreationRect.bottom = CreationRect.top + 100;

	CreationOffset.y = YOffset += 40;

	// make the select mode modes
	g3_SelectMode1.Create( WS_CHILD|WS_VISIBLE |CBS_DROPDOWNLIST |WS_VSCROLL,CreationRect,this,SINUSOIDALSELECTMODE1);
	g3_SelectMode1.AddString("Cos(Distance(Dx2,Dz2))");
	g3_SelectMode1.AddString("Cos(Dx)*Cos(Dz)");
	g3_SelectMode1.AddString("Cos(Dx)+Cos(Dz)");
	g3_SelectMode1.AddString("Sin(Distance(Dx2,Dz2))");
	g3_SelectMode1.AddString("Sin(Dx)*Sin(Dz)");
	g3_SelectMode1.AddString("Sin(Dx)+Sin(Dz)");
	RollOutManager.AddControltoGroup((CWnd *)&g3_SelectMode1,CreationRect);	
	g3_SelectMode1.SetFont(Fonts.GetFont(MSSANSSERIF));
	g3_SelectMode1.SetCurSel(0);

	CreationRect.left = 20;
	CreationRect.right = 180;
	CreationRect.top = YOffset;
	CreationRect.bottom = CreationRect.top + 100;

	g3_SelectMode2.Create( WS_CHILD|WS_VISIBLE |CBS_DROPDOWNLIST |WS_VSCROLL,CreationRect,this,SINUSOIDALSELECTMODE2);
	g3_SelectMode2.AddString("Hill");
	g3_SelectMode2.AddString("Ripple W/Max Radius");
	g3_SelectMode2.AddString("Ripple W/LinDecreasing Radius");
	g3_SelectMode2.AddString("Hill Absolute");
	RollOutManager.AddControltoGroup((CWnd *)&g3_SelectMode2,CreationRect);	
	g3_SelectMode2.SetFont(Fonts.GetFont(MSSANSSERIF));
	g3_SelectMode2.SetCurSel(0);

	// load up the terrain functions array for later use
	TerrainFunctions[0][0] = Cos3Da;
	TerrainFunctions[0][1] = Cos3DRa;
	TerrainFunctions[0][2] = Cos3DLDRa;
	TerrainFunctions[1][0] = Cos3Db;
	TerrainFunctions[1][1] = Cos3DRb;
	TerrainFunctions[1][2] = Cos3DLDRb;
	TerrainFunctions[2][0] = Cos3Dc;
	TerrainFunctions[2][1] = Cos3DRc;
	TerrainFunctions[2][2] = Cos3DLDRc;
	TerrainFunctions[3][0] = Sin3Da;
	TerrainFunctions[3][1] = Sin3DRa;
	TerrainFunctions[3][2] = Sin3DLDRa;
	TerrainFunctions[4][0] = Sin3Db;
	TerrainFunctions[4][1] = Sin3DRb;
	TerrainFunctions[4][2] = Sin3DLDRb;
	TerrainFunctions[5][0] = Sin3Dc;
	TerrainFunctions[5][1] = Sin3DRc;
	TerrainFunctions[5][2] = Sin3DLDRc;

	CreationOffset.y = YOffset += 30;

	// create the height constant spin edit control
	g3_HeightConstant.Create("Height Constant:",CreationOffset,0,Info.Height,50,SE_FLOAT | SE_RIGHTJUSTIFIED,this,SINUSOIDALHEIGHTCONSTANT);
	g3_HeightConstant.SetValue(1.0f);
	// adds its callback function
	CallbackFunctions[SINUSOIDALHEIGHTCONSTANT] = On_G3HeightConstant;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g3_HeightConstant,g3_HeightConstant.GetWindowRect());

	CreationOffset.y = YOffset += 25;
	
	// create the Period value spin edit control
	g3_Period.Create("Period:",CreationOffset,0,20,200,SE_FLOAT | SE_RIGHTJUSTIFIED,this,SINUSOIDALPERIOD);
	g3_Period.SetValue(3.0f);
	// adds its callback function
	CallbackFunctions[SINUSOIDALPERIOD] = On_G3Period;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g3_Period,g3_Period.GetWindowRect());	

	CreationOffset.y = YOffset += 25;

	// now add the LDRradius
	g3_LDRadius.Create("LDRadius: ",CreationOffset,0,20,200,SE_FLOAT | SE_RIGHTJUSTIFIED,this,SINUSOIDALRADIUS);
	g3_LDRadius.SetValue(3.0f);
	// adds its callback function
	CallbackFunctions[SINUSOIDALRADIUS] = On_G3LDRadius;
	// add it to the roll out
	RollOutManager.AddControltoGroup((CWnd *)&g3_LDRadius,g3_LDRadius.GetWindowRect());	
}
